package com.pica.cloud.account.account.server.util;

import com.alibaba.fastjson.JSONObject;
import com.pica.cloud.account.account.server.entity.Account;
import com.pica.cloud.account.account.server.entity.Doctor;
import com.pica.cloud.account.account.server.enums.SourceTypeEnum;
import com.pica.cloud.account.account.server.mapper.DoctorMapper;
import com.pica.cloud.account.account.server.mapper.LogLoginMapper;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.StringUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Token工具类
 */
@Component
public class TokenUtils {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICacheClient cacheClient;

    @Autowired
    private DoctorMapper doctorMapper;

    @Autowired
    private LogLoginMapper logLoginMapper;
    /**
     * 校验token的状态
     *
     * @param token
     * @return
     */
    public boolean checkTokenStatus(String token) {
        String str = cacheClient.get("token-" + token);
        return StringUtils.isBlank(str);
    }

    /**
     * @Description token续时
     * @Author Chongwen.jiang
     * @Date 2020/2/26 13:07
     * @ModifyDate 2020/2/26 13:07
     * @Params [token, sourceType]
     * @Return boolean
     */
    public boolean tokenContinueTime(String token, Integer sourceType, Integer seconds) {
        logger.info("tokenContinueTime-token-{}, sourceType-{}, seconds-{}", token, sourceType, seconds);

        //  非游客token才去执行token续时代码
        Map<String, String> map = cacheClient.getToken(token);
        if (!CollectionUtils.isEmpty(map)) {
            String id = map.get("id");
            logger.info("tokenContinueTime-start-userId-{}", id);
            if (StringUtils.isNotEmpty(id) && Integer.parseInt(id) > 0) {
                // 根据不同的产品线设置token有效期（web/admin token有效期30天，ios/android/h5 token有效期24小时）
                int expiredSeconds = 24 * 60 * 60;
                if (!SourceTypeEnum.SAAS.getCode().equals(sourceType) &&
                        !SourceTypeEnum.ADMIN.getCode().equals(sourceType)) {
                    expiredSeconds = expiredSeconds * 30;
                }

                String sourceTypeRedis = AccountUtils.getSourceType(sourceType);
                String key = "token-" + token;
                //  value = "token-doctor-{doctorId}";
                if (!cacheClient.exists(key)) {
                    logger.info("tokenContinueTime-key not exists");
                    return false;
                }
                String value = cacheClient.get(key);
                if (!cacheClient.exists(value + "-" + sourceTypeRedis)) {
                    logger.info("tokenContinueTime-value not exists");
                    return false;
                }
                try {
                    cacheClient.expire(key, expiredSeconds);
                    logger.info("tokenContinueTime-{} key: {} value: {}", id, key, value);
                    cacheClient.expire(value + "-" + sourceTypeRedis, expiredSeconds);
                    return true;
                } catch (Exception e) {
                    logger.error("tokenContinueTime-Exception-{}" + e.getMessage(), e);
                    return false;
                }
            }
        }
        return false;
    }

    /**
     * 获取新的token
     *
     * @param account
     * @return
     */
    public String generateToken(Account account) {
        logger.info("generateToken account:{}",JSONObject.toJSONString(account));
        Integer registerSource = account.getRegisterSource();
        String newToken = "";
        try {
            //先清除旧token
            String value = "token-doctor-" + account.getId().toString();
            String sourceType = AccountUtils.getSourceType(registerSource);
            String oldToken = cacheClient.get(value + "-"+sourceType);
            if (StringUtils.isNotBlank(oldToken)) {
                Long del = cacheClient.del(oldToken);
                logger.info("oldToken: {} num: {}", oldToken, del);

                // log_login 中查询出所有的token，并删除
                List<String> tokenList =  logLoginMapper.selectTokenByAcctId(account.getAcctId());
                if (StringUtil.isNotEmptyList(tokenList)) {
                    for (int i = 0 ; i < tokenList.size(); i++) {
                        String tokenTmp = "token-" + tokenList.get(i);
                        if (StringUtils.isNotBlank(cacheClient.get(tokenTmp))) {
                            del = cacheClient.del(tokenTmp);
                            logger.info("oldToken: {} num: {} i：{}", tokenTmp, del, i);
                        } else {
                            logger.info("oldToken: {} 不存在 i：{}", tokenTmp, i);
                        }
                    }
                }
            }
            // 根据不同的产品线设置token有效期
            int expiredSeconds = 24 * 60 * 60;
            if (registerSource != 3 && registerSource != 5) {
                expiredSeconds = expiredSeconds * 30;
            }
            newToken = UUID.randomUUID().toString().replace("-", "").toUpperCase();
            String Key = "token-" + newToken;
            //存储token对应的用户id，数据结构如：(token-FF9FCB0D93A642328A01C279701B7607：token-doctor-1)
            cacheClient.set(Key, value, expiredSeconds);
            logger.info("generateToken-{} key: {} value: {}", account.getId(), Key, value);
            //存储不同终端对应的token，数据结构如：(token-doctor-12345678-app：token-FF9FCB0D93A642328A01C279701B7607)
            //String sourceType = AccountUtils.getSourceType(registerSource);
            cacheClient.set(value + "-" + sourceType, Key, expiredSeconds);
            Map<String, String> data = new HashMap<>();
            data.put("token", newToken);
            data.put("id", account.getId() + "");
            data.put("acctId", account.getAcctId() + "");
            if (!StringUtils.isBlank(account.getMobilePhone())) {
                data.put("mobile", AESUtil.encryptV0(account.getMobilePhone()));
            } else {
                data.put("mobile", "");
            }
            data.put("name", account.getMobilePhone().replaceAll("(\\d{3})\\d{4}(\\w{4})", "$1****$2"));
            data.put("created_time", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(account.getCreatTime()));
            data.put("sysCode", sourceType);
            if (account.getId() != null) {
                Doctor doctor = doctorMapper.selectByPrimaryKey(account.getId().intValue());
                data.put("avatar_image_url", doctor.getAvatarImageUrl() + "");
                data.put("town_id", doctor.getTown() + "");
                data.put("town_name", doctor.getTownName() + "");
                data.put("county_id", doctor.getCounty() + "");
                data.put("county_name", doctor.getCountyName() + "");
                data.put("city_id", doctor.getCity() + "");
                data.put("city_name", doctor.getCityName() + "");
                data.put("province_id", doctor.getProvince() + "");
                data.put("province_name", doctor.getProvinceName() + "");
                data.put("hospital", doctor.getHospital() + "");
                data.put("hospital_id", doctor.getHospitalId() + "");
                data.put("department", doctor.getDepartment() + "");
                data.put("department_id", doctor.getDepartmentId() + "");
                data.put("title", doctor.getTitle() + "");
                data.put("title_id", doctor.getTitleId() + "");
                data.put("status", doctor.getStatus() + "");
                if (!StringUtils.isBlank(doctor.getName())){
                    data.put("name", doctor.getName() + "");
                }
            }
            Iterator<Map.Entry<String, String>> iterator = data.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, String> map = iterator.next();
                String key = map.getKey();
                String valueInfo = map.getValue();
                //存储token：(token-doctor-1：用户数据)
                cacheClient.hset(value, key, valueInfo);
            }
//            } else {
//                cacheClient.hset(value, "token", newToken);
//            }
        } catch (Exception ex) {
            logger.error("生成token异常：{}",ex.getMessage());
        }
        return newToken;
    }
}
