package com.pica.cloud.account.account.server.controller;

import com.pica.cloud.account.account.server.entity.Account;
import com.pica.cloud.account.account.server.entity.EncryptEntity;
import com.pica.cloud.account.account.server.mapper.AccountMapper;
import com.pica.cloud.account.account.server.req.AccountReq;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.service.AccountService;
import com.pica.cloud.account.account.server.service.CaptchaService;
import com.pica.cloud.account.account.server.util.CryptoUtil;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.constants.CommonConstants;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.foundation.utils.utils.CommonUtil;
import com.pica.cloud.foundation.utils.utils.ValidateUtils;
import com.pica.cloud.foundation.utils.utils.json.Object2Map;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.Map;
import java.util.UUID;

/**
 * 账号相关controller
 *
 * @author andong
 * @create 2019/5/20
 */
@Api(description = "账号管理")
@RestController
@RequestMapping("/account")
public class AccountController extends AccountBaseController {

    private final String AUTH_CODE_PREFIX = "authCode-";
    private Logger logger = LoggerFactory.getLogger(AccountController.class);
    @Autowired
    private AccountService accountService;

    @Autowired
    @Qualifier("cacheMigrateClient")
    private ICacheClient redisClient;

    @GetMapping("/test")
    public String test() {
        return "test";
    }


    @ApiOperation("微信登录")
    @PostMapping("/login/wechat")
    public PicaResponse<String> wechatLogin(@RequestBody AccountReq req) {
        if (StringUtils.isBlank(req.getUnionid())) {
            return PicaResponse.toResponse(null, PicaResultCode.PARAM_IS_INVALID.code(), "缺少unionid");
        }

        Account account = accountService.getByUnionid(req.getUnionid());  //获取账号信息
        if (account == null) {
            return PicaResponse.toResponse(null, PicaResultCode.RESULE_DATA_NONE.code(), "该unionid未绑定云鹊医账号");
        }

        //已经绑定过云鹊医账号，登录成功，返回token
        String deviceType = super.getDeviceInfo("device_type");  //1:pc 2:android 3:ios
        String newToken = this.generateToken(account, deviceType);
        return PicaResponse.toResponse(newToken);
    }

    //    @ApiOperation("密码或验证码登录")
//    @PostMapping("/login")
    public PicaResponse<String> login(@RequestBody AccountReq req) {
        this.checkMobilePhone(req.getMobilePhone());
        Account account = accountService.getByMobilePhone(req.getMobilePhone());  //获取账号信息
        if (account == null) {
            return PicaResponse.toResponse(null, PicaResultCode.RESULE_DATA_NONE.code(), "未注册，请验证码登录");
        }

        if (StringUtils.isBlank(req.getPassword())) {  //验证码登录
            req.setFlag("0");
            this.checkAuthCode(req);  //校验验证码
        } else {  //密码登录
            if (!StringUtils.equals(req.getPassword(), account.getPassword())) {
                return PicaResponse.toResponse(null, PicaResultCode.PARAM_IS_INVALID.code(), "请输入正确的密码");
            }
        }

        //更新最后登录时间
        Account update = new Account();
        update.setId(account.getId());
        update.setLastLoginTime(new Date());
        if (StringUtils.isBlank(account.getUnionid()) && StringUtils.isNotBlank(req.getUnionid())) {
            update.setUnionid(req.getUnionid());  //绑定微信unionid
            update.setModifyId(account.getId());
            update.setModifyTime(new Date());
        }
        accountService.updateAccountById(update);

        //登录成功，清除旧token，生成新token
        String deviceType = super.getDeviceInfo("device_type");  //1:pc 2:android 3:ios
        String newToken = this.generateToken(account, deviceType);
        return PicaResponse.toResponse(newToken);
    }

    //    @ApiOperation("注册")
//    @PostMapping("/register")
    public PicaResponse<String> register(@RequestBody AccountReq req) {
        this.checkMobilePhone(req.getMobilePhone());
        this.checkAuthCode(req);
        String deviceType = super.getDeviceInfo("device_type");  //1:pc 2:android 3:ios
        Account account = new Account();
        account.setMobilePhone(req.getMobilePhone());
        switch (deviceType) {  //注册来源
            case "1":
                account.setRegisterSource(CommonConstants.SYSTEM_TYPE_P024_NO_3);  //pc
                break;
            case "2":
                account.setRegisterSource(CommonConstants.SYSTEM_TYPE_P024_NO_1);  //android
                break;
            case "3":
                account.setRegisterSource(CommonConstants.SYSTEM_TYPE_P024_NO_2);  //ios
                break;
            default:
                account.setRegisterSource(7);  //H5注册
        }
        //创建账号
        accountService.createAccount(account);

        //生成token并返回
        String newToken = this.generateToken(account, deviceType);
        return PicaResponse.toResponse(newToken);
    }

    @ApiOperation("H5端一键登录功能,无需完善信息")
    @PostMapping("/login-register")
    public PicaResponse<String> loginRegister(@RequestBody AccountReq req) {
        this.checkMobilePhone(req.getMobilePhone());
        req.setPassword(null);  //登录或注册，只能使用验证码

        //判断账号是否已经存在
        Account account = accountService.getByMobilePhone(req.getMobilePhone());
        if (account != null) {
            return this.login(req);  //登录
        } else {
            return this.register(req);  //注册
        }
    }

    //手机格式校验
    private void checkMobilePhone(String mobilePhone) {
        if (StringUtils.isBlank(mobilePhone) || !ValidateUtils.isMobile(mobilePhone)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "请输入正确的手机号");
        }
    }

    //获取验证码redis key
    private String getAuthCodeKey(String mobilePhone, String flag) {
        return AUTH_CODE_PREFIX + flag + "-" + EncryptUtils.encryptContent(mobilePhone, EncryptConstants.ENCRYPT_TYPE_MOBILE);
    }

    //校验验证码
    private void checkAuthCode(AccountReq req) {
        String flag = StringUtils.isBlank(req.getFlag()) ? "0" : req.getFlag();

        if (StringUtils.isBlank(req.getAuthCode())) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "短信验证码错误");
        }

        String authCodeKey = this.getAuthCodeKey(req.getMobilePhone(), flag);
        String cacheCode = redisClient.get(authCodeKey);  //从redis获取验证码
        if (StringUtils.isBlank(cacheCode)) {
            throw new PicaException(PicaResultCode.RESULE_DATA_NONE.code(), "短信验证码已过期，请重新获取");
        }
        if (!StringUtils.equals(req.getAuthCode(), cacheCode)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "短信验证码错误");
        }
        redisClient.del(authCodeKey);  //清除验证码
    }

    //生成H5 token
    private String generateH5Token(Account account) {
        String newToken = StringUtils.EMPTY;
        try {
            //先清除旧token
            String tokenValue = "token-doctor-" + account.getId().toString();
            String oldToken = redisClient.get(tokenValue + "-h5");
            if (StringUtils.isNotBlank(oldToken)) {
                redisClient.del(oldToken);
            }
            //生成新token
            int expiredSeconds = 30 * 24 * 60 * 60;  //H5 token有效期30天
            newToken = UUID.randomUUID().toString().replace("-", "").toUpperCase();
            String tokenKey = "token-" + newToken;
            redisClient.set(tokenKey, tokenValue, expiredSeconds);
            redisClient.set(tokenValue + "-h5", tokenKey, expiredSeconds);
            //用户数据放入缓存
            String userData = redisClient.hget(tokenValue, "id");
            if (StringUtils.isEmpty(userData)) {
                PicaUser picaUser = new PicaUser();
                picaUser.setToken(newToken);
                picaUser.setId(account.getId().intValue());
                picaUser.setMobile(account.getMobilePhone());
                picaUser.setName(EncryptUtils.decryptContent(account.getMobilePhone(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY).replaceAll("(\\d{3})\\d{4}(\\w{4})", "$1****$2"));
                picaUser.setCreated_time(account.getCreatTime());
                Map<String, String> data = Object2Map.objectToMapString("yyyy-MM-dd HH:mm:ss", picaUser, new String[0]);
                data.put("sysCode", "h5");
                data.forEach((key, value) -> {
                    value = value == null ? "" : value;
                    redisClient.hset(tokenValue, key, value);
                });
            }
        } catch (Exception ex) {
            logger.error("生成H5 token异常：{}" + ex.getMessage(), ex);
        }
        return newToken;
    }

    //清除旧token，生成新token
    private String generateToken(Account account, String deviceType) {
        String newToken;
        switch (deviceType) {  //设备信息
            case "1":  //pc
                //TODO
                newToken = StringUtils.EMPTY;
                break;
            case "2":  //android
            case "3":  //ios
                //TODO  更新设备信息
                newToken = StringUtils.EMPTY;
                break;
            default:  //H5
                newToken = this.generateH5Token(account);
        }
        return newToken;
    }

}
