package com.pica.cloud.account.account.server.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.entity.AccountInfoEntity;
import com.pica.cloud.account.account.server.entity.Doctor;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.exception.AccountException;
import com.pica.cloud.account.account.server.mapper.*;
import com.pica.cloud.account.account.server.req.DoctorReq;
import com.pica.cloud.account.account.server.service.DoctorService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.AccountUtils;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaWarnException;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.StringUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Created on 2019/9/11 11:02
 * author:crs
 * Description:医生相关Service
 */
@Service
public class DoctorServiceImpl implements DoctorService {

    @Autowired
    private DoctorMapper doctorMapper;

    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;

    @Autowired
    private DoctorEducationMapper doctorEducationMapper;

    @Autowired
    private DoctorHospitalMapper doctorHospitalMapper;

    @Autowired
    private AccountUnionMapper accountUnionMapper;

    @Autowired
    private ICacheClient cacheClient;

    @Override
    public Doctor getDoctorInfo(Integer id) {
        return doctorMapper.selectByPrimaryKey(id);
    }

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Transactional
    @Override
    public void modifyDoctorInfo(Doctor doctor, Long doctorId) {
        Integer id = doctor.getId();
        if (id == null) {
            throw new AccountException(AccountExceptionEnum.PICA_PARAMS_ERROR);
        }
        Doctor entity = doctorMapper.selectByPrimaryKey(id);
        if (entity == null) {
            throw new AccountException(AccountExceptionEnum.PICA_NOT_EXIST);
        }
        String mobilePhone = doctor.getMobilePhone();
        if (!StringUtils.isBlank(mobilePhone)) {
            AccountUtils.checkMobilePhone(doctor.getMobilePhone());
        }
        doctor.setMobilePhone(AESUtil.encryptV0(mobilePhone));
        doctor.setModifyTime(new Date());
        if (StringUtils.isNotBlank(doctor.getCard())) {
            doctor.setCard(EncryptUtils.encryptContent(doctor.getCard(), EncryptConstants.ENCRYPT_TYPE_DOCTOR_IDNO));
        }
        doctorMapper.updateByPrimaryKeySelective(doctor);
        Integer acctId = entity.getAcctId();
        AccountInfoEntity accountInfoEntity = new AccountInfoEntity();
        accountInfoEntity.setMobilePhone(AESUtil.encryptV0(mobilePhone));
        accountInfoEntity.setId(acctId);
        accountInfoEntity.setModifiedId(doctorId.intValue());
        accountInfoEntity.setModifiedTime(new Date());
        accountInfoDetailMapper.updateByPrimaryKeySelective(accountInfoEntity);
    }

    @Override
    public int registerWelcome(DoctorReq doctorReq) {
        logger.info("registerWelcome req:{}",JSONObject.toJSONString(doctorReq));
        Doctor doctor = doctorMapper.selectByPrimaryKey(doctorReq.getId());
        if (null == doctor) {
            throw new PicaWarnException(AccountExceptionEnum.PICA_PARAMS_ERROR.getCode(),
                    AccountExceptionEnum.PICA_PARAMS_ERROR.getMessage());
        }
        if(StringUtil.isNotNull(doctorReq.getDepartment())){
            doctor.setDepartment(doctorReq.getDepartment());
        }
        if(StringUtil.isNotNull(doctorReq.getTitle())){
            doctor.setTitle(doctorReq.getTitle());
        }
        if(StringUtil.isNotNull(doctorReq.getName())){
            doctor.setName(doctorReq.getName());
        }
        doctor.setModifyTime(new Date());
        doctor.setModifyId(doctorReq.getId());
        return doctorMapper.updateByPrimaryKeySelective(doctor);
    }

    @Transactional
    @Override
    public void deleteDoctorInfo(Integer id, Long modifyId) {
        //p_doctor表修改记录状态
        Doctor doctor = doctorMapper.selectByPrimaryKey(id);
        if (doctor != null) {
            Integer acctId = doctor.getAcctId();
            HashMap<String, Long> map = new HashMap<>();
            map.put("id", id.longValue());
            map.put("modifyId", modifyId);
            doctorMapper.updateDeleteByPrimaryKey(map);
            HashMap<String, Long> acctMap = new HashMap<>();
            acctMap.put("modifyId", modifyId);
            acctMap.put("acctId", acctId.longValue());
            accountInfoDetailMapper.updateDeleteByPrimaryKey(acctMap);
            logger.info("deleteDoctorInfo acctId:{}", acctId);
            accountUnionMapper.deleteByAcctId(acctId);
        } else {
            throw new AccountException(AccountExceptionEnum.PICA_NOT_REGISTER);
        }
    }

    /**
     * 客服管理后台添加医生
     * 1)当前医生信息是否存在,如果已经存在直接抛出异常；
     * 2)账户表和医生表；
     * 3)手机号进行加密存储；
     * 4)账户基本信息的处理:姓名,性别,年龄,生日,身份证号；
     *
     * @param doctor 新的医生数据
     */
    @Override
    @Transactional
    public Integer addDoctorInfo(Doctor doctor, Long doctorId) {
        String mobilePhone = doctor.getMobilePhone();
        String mobileEncrypt = AESUtil.encryptV0(mobilePhone);
        AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(mobileEncrypt);
        if (accountInfoEntity != null) {
            throw new AccountException(AccountExceptionEnum.PICA_ALREADY_REGISTER);
        }
        String password = doctor.getPassword();
        Date currentTime = new Date();
        AccountInfoEntity entity = new AccountInfoEntity();
        entity.setMobilePhone(mobileEncrypt);
        entity.setPassword(password);
        entity.setCreatedTime(currentTime);
        entity.setModifiedTime(currentTime);
        entity.setModifiedId(doctorId.intValue());
        entity.setRegisterProduct(AccountTypeEnum.PRODUCT_TYPE_DOCTOR.getCode());
        entity.setRegTime(currentTime);
        //客服系统新增医生
        entity.setRegisterSource(AccountTypeEnum.DEVICE_TYPE_ADMIN.getCode());
        entity.setCreatedId(doctorId.intValue());
        entity.setDeleteFlag(1);
        entity.setSex(doctor.getSex());
        entity.setName(doctor.getName());
        doctor.setCard(EncryptUtils.encryptContent(doctor.getCard(), EncryptConstants.ENCRYPT_TYPE_DOCTOR_IDNO));
        entity.setIdCard(doctor.getCard());
        entity.setAge(doctor.getAge());
        entity.setBirthday(doctor.getBirthday());
        accountInfoDetailMapper.insertSelective(entity);
        doctor.setAcctId(entity.getId());
        doctor.setMobilePhone(mobileEncrypt);
        doctor.setModifyId(doctorId.intValue());
        doctor.setCreatId(doctorId.intValue());
        doctor.setCreatTime(currentTime);
        doctor.setModifyTime(new Date());
        doctor.setRegisterSource(AccountTypeEnum.DEVICE_TYPE_ADMIN.getCode());
        doctor.setDeleteFlag(1);
        doctorMapper.insertSelective(doctor);
        return doctor.getId();
    }

    @Override
    @Transactional
    public void bindWeChat(int acctId, String unionId) {
        doctorMapper.bindWeChat(acctId, unionId);
    }

    @Override
    @Transactional
    public void unbindWeChat(int acctId) {
        doctorMapper.unbindWeChat(acctId);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void revertAccts(List<Integer> doctorIds) {
        logger.info("revertAccts:{}", JSONObject.toJSONString(doctorIds));
        if (doctorIds.size() == 0) {
            return;
        }

        //check docIds 是否都是已经删除了
        List<Doctor> docList = doctorMapper.getRevertListByDoctorIds(doctorIds);
        if (docList.size() != doctorIds.size()) {
            logger.error("revertAccts: 入参中存在未删除的医生id");
            return;
        }

        //revert doctor数据
        doctorMapper.updateRevertByIdList(doctorIds);

        //revert account数据
        List<Integer> acctIds = docList.stream().map(obj -> obj.getAcctId()).collect(Collectors.toList());
        accountInfoDetailMapper.updateRevertByAcctIds(acctIds);

        //revert doctor 教育信息
        doctorEducationMapper.updateRevertByDocIds(doctorIds);

        //revert 医生机构
        doctorHospitalMapper.updateRevertByDocIds(doctorIds);
        logger.info("revertAccts:end");
    }


    @Override
    public List<Integer> getDoctorsByInnerOrg() {
        String ids = cacheClient.get(Constants.INNER_ORG_IDS);
        if (StringUtil.isEmpty(ids)) {
            List<Integer> dbIds = doctorMapper.getDoctorsByInnerOrg(null);
            String arrStr = "";
            for (Integer i : dbIds) {
                arrStr = arrStr + i + ",";
            }
            cacheClient.setex(Constants.INNER_ORG_IDS, arrStr, 3600);
            return dbIds;
        } else {
            List<Integer> rtnIds = Arrays.asList(ids.split(",")).stream().filter(s -> !"".equals(s)).map(s -> Integer.parseInt(s)).collect(Collectors.toList());
            return rtnIds;
        }
    }


}

