package com.pica.cloud.account.account.server.controller;


import com.pica.cloud.account.account.server.entity.EncryptEntity;
import com.pica.cloud.account.account.server.entity.LogLoginEntity;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.exception.AccountException;
import com.pica.cloud.account.account.server.log.AccountLogEntityUtils;
import com.pica.cloud.account.account.server.log.AccountLogUtils;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.service.LoginService;
import com.pica.cloud.account.account.server.service.TokenService;
import com.pica.cloud.account.account.server.util.AccountUtils;
import com.pica.cloud.account.account.server.util.CryptoUtil;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.redis.ICacheClient;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

@Api(description = "登录资源")
@RestController
public class LoginController extends AccountBaseController {

    @Autowired
    private LoginService loginService;

    @Autowired
    @Qualifier("cacheMigrateClient")
    private ICacheClient redisClient;

    @Autowired
    private AccountLogUtils picaLogUtils;

    @Autowired
    private AccountUtils accountUtils;

    @Autowired
    private TokenService tokenService;

    /**
     * 密码登录接口
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @ApiOperation("密码登录接口")
    @PostMapping("/login")
    public PicaResponse<String> loginByPassword(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setProductType(super.getProductType());
        request.setSourceType(super.getSourceType());
        request.setLoginIp(super.getIpAddr());
        AccountUtils.checkMobilePhone(request.getMobile());
        AccountUtils.checkPassword(request.getPassword());
        String result = loginService.login(request);
        return PicaResponse.toResponse(result);
    }

    /**
     * 一键登录
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @ApiOperation("一键登录接口")
    @PostMapping(value = "/login-register")
    public PicaResponse loginAndRegister(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        accountUtils.checkMobilePhoneAndAuthCode(request.getMobile(), AccountTypeEnum.SYSCODE_TYPE_LOGIN.getCode() + "",request.getAuthCode());
        request.setProductType(super.getProductType());
        request.setSourceType(super.getSourceType());
        request.setLoginIp(super.getIpAddr());
        String json = loginService.loginAndRegister(request);
        return PicaResponse.toResponse(json);
    }

    @ApiOperation("微信登录接口")
    @PostMapping(value = "/login/wechat")
    public PicaResponse loginByWeChat(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setProductType(super.getProductType());
        request.setSourceType(super.getSourceType());
        request.setLoginIp(super.getIpAddr());
        String result = loginService.loginByWeChat(request);
        return PicaResponse.toResponse(result);
    }

    @ApiOperation("微信登录第二步接口")
    @PostMapping(value = "/login/wechat/step2")
    public PicaResponse loginByWeChatStep(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        accountUtils.checkMobilePhoneAndAuthCode(request.getMobile(), AccountTypeEnum.SYSCODE_TYPE_WE_CHAT.getCode() + "", request.getAuthCode());
        request.setProductType(super.getProductType());
        request.setSourceType(super.getSourceType());
        request.setLoginIp(super.getIpAddr());
        String result = loginService.loginByWeChatStep(request);
        return PicaResponse.toResponse(result);
    }

    /**
     * 绑定微信接口
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @ApiOperation("绑定微信接口")
    @PostMapping("/login/wechat/bind")
    public PicaResponse<String> bindWeChat(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setAccId(super.getAcctId());
        loginService.bindWeChat(request);
        return PicaResponse.toResponse();
    }

    @ApiOperation("微信解除绑定接口")
    @PutMapping("/login/wechat/unbind")
    public PicaResponse unbindWeChat() {
        Integer acctId = super.getAcctId();
        loginService.unbindWeChat(acctId);
        return PicaResponse.toResponse();
    }

    /**
     * 退出登录接口，返回一个随机token
     * 1)登录状态调用，2)清除token
     *
     * @return
     */
    @ApiOperation(value = "退出登录接口")
    @GetMapping("/logout")
    public PicaResponse loginOut() {
        String token = super.getToken();
        if (StringUtils.isNotEmpty(token)) {
            Integer acctId = super.getAcctId();
            redisClient.deleteToken(token);
            Map<String, Object> headersMap = super.getHeaders();
            String newToken = tokenService.getToken(headersMap);
            LogLoginEntity entity = AccountLogEntityUtils.getLogLoginEntity(acctId, super.getProductType(), super.getSourceType(),
                    AccountTypeEnum.LOGIN_OUT.getCode(), super.getIpAddr(), AccountTypeEnum.LOGIN_STATUS_SUCCESS.getCode(), AccountTypeEnum.LOG_TYPE_LOGIN.getCode());
            picaLogUtils.info(entity);
            return PicaResponse.toResponse(newToken);
        } else {
            throw new AccountException(AccountExceptionEnum.PICA_LOGIN_AGAIN);
        }
    }
}
