package com.pica.cloud.account.account.server.controller;

import com.pica.cloud.account.account.server.entity.*;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.exception.AccountException;
import com.pica.cloud.account.account.server.log.AccountLogEntityUtils;
import com.pica.cloud.account.account.server.log.AccountLogUtils;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.req.OneClickLoginReq;
import com.pica.cloud.account.account.server.service.DoctorService;
import com.pica.cloud.account.account.server.service.LoginService;
import com.pica.cloud.account.account.server.service.TokenService;
import com.pica.cloud.account.account.server.util.AccountUtils;
import com.pica.cloud.account.account.server.util.CryptoUtil;
import com.pica.cloud.account.account.server.vo.OneClickLoginResultVo;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.redis.CacheClient;
import com.pica.cloud.foundation.redis.ICacheClient;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;

@Api(description = "登录资源")
@RestController
public class LoginController extends AccountBaseController {

    @Autowired
    private LoginService loginService;

    @Autowired
    private ICacheClient redisClient;

    @Autowired
    private AccountLogUtils picaLogUtils;

    @Autowired
    private AccountUtils accountUtils;

    @Autowired
    private TokenService tokenService;

    @Autowired
    private DoctorService doctorService;

    @Autowired
    private CacheClient cacheClient;

    private String cache_prifix = "cache-";

    /**
     * 密码登录接口
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @ApiOperation("密码登录接口")
    @PostMapping("/login")
    public PicaResponse<LoginResult> loginByPassword(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setProductType(super.getProductType());
        Integer sourceType = super.getSourceType();
        request.setSourceType(sourceType);
        request.setLoginIp(super.getIpAddr());
        AccountUtils.checkMobilePhone(request.getMobile());
        AccountUtils.checkPassword(request.getPassword());
        LoginResult login = loginService.login(request);
        if (sourceType == 3) {
            login.setDoctorId("");
        } else {
            login.setUserId(null);
        }
        return PicaResponse.toResponse(login);
    }

    /**
     * 一键登录
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @ApiOperation("PC,App端一键登录接口，需要完善信息")
    @PostMapping(value = "/login-register")
    public PicaResponse<LoginResult> loginAndRegister(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setProductType(super.getProductType());
        Integer sourceType = super.getSourceType();
        request.setSourceType(sourceType);
        request.setLoginIp(super.getIpAddr());
        LoginResult login = loginService.loginAndRegister(request);
        if (sourceType == 3) {
            login.setDoctorId("");
        } else {
            login.setUserId(null);
        }
        return PicaResponse.toResponse(login);
    }

    @ApiOperation("微信登录接口")
    @PostMapping(value = "/login/wechat")
    public PicaResponse<LoginResult> loginByWeChat(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setProductType(super.getProductType());
        request.setSourceType(super.getSourceType());
        request.setLoginIp(super.getIpAddr());
        LoginResult result = loginService.loginByWeChat(request);
        return PicaResponse.toResponse(result);
    }

    @ApiOperation("微信登录第二步接口")
    @PostMapping(value = "/login/wechat/step2")
    public PicaResponse<LoginResult> loginByWeChatStep(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        accountUtils.checkMobilePhoneAndAuthCode(request.getMobile(), AccountTypeEnum.SYSCODE_TYPE_WE_CHAT.getCode() + "", request.getAuthCode());
        request.setProductType(super.getProductType());
        request.setSourceType(super.getSourceType());
        request.setLoginIp(super.getIpAddr());
        LoginResult result = loginService.loginByWeChatStep(request);
        return PicaResponse.toResponse(result);
    }

    /**
     * 绑定微信接口
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @ApiOperation("绑定微信接口")
    @PostMapping("/login/wechat/bind")
    public PicaResponse bindWeChat(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);

        Long result = cacheClient.setnx(cache_prifix + request.getWeChatCode(), request.getWeChatCode());
        if (result == 1) {
            cacheClient.set(cache_prifix + request.getWeChatCode(), 60);
            long doctorId = super.getDoctorIdByToken();
            String nickname = loginService.bindWeChat(doctorId, request);
            Map<String, String> map = new HashMap();
            map.put("nickname", nickname);
            //成功以后释放锁
            cacheClient.del(cache_prifix + request.getWeChatCode());
            return PicaResponse.toResponse(map);
        } else {
            cacheClient.del(cache_prifix + request.getWeChatCode());
            return PicaResponse.toResponse(null, AccountExceptionEnum.PICA_WECHAT_UNBIND_CURRENT.getCode(),
                    AccountExceptionEnum.PICA_WECHAT_UNBIND_CURRENT.getMessage());
        }
    }

    @ApiOperation("微信解除绑定接口")
    @PutMapping("/login/wechat/unbind")
    public PicaResponse unbindWeChat() {
        long doctorId = super.getDoctorIdByToken();
        loginService.unbindWeChat(doctorId);
        return PicaResponse.toResponse();
    }

    /**
     * 退出登录接口，返回一个随机token
     * 1)登录状态调用，2)清除token
     *
     * @return
     */
    @ApiOperation(value = "退出登录接口")
    @GetMapping("/logout")
    public PicaResponse<String> loginOut() {
        String token = super.getToken();
        if (StringUtils.isNotEmpty(token)) {
            Long doctorId = super.getDoctorIdByToken();
            Doctor doctorInfo = doctorService.getDoctorInfo(doctorId.intValue());
            Integer acctId = doctorInfo.getAcctId();
            redisClient.deleteToken(token);
            Map<String, Object> headersMap = super.getHeaders();
            String newToken = tokenService.getToken(headersMap);
            LogLoginEntity entity = AccountLogEntityUtils.getLogLoginEntity(acctId, super.getProductType(), super.getSourceType(),
                    AccountTypeEnum.LOGIN_OUT.getCode(), super.getIpAddr(), AccountTypeEnum.LOGIN_STATUS_SUCCESS.getCode(), AccountTypeEnum.LOG_TYPE_LOGIN.getCode());
            picaLogUtils.info(entity);
            return PicaResponse.toResponse(newToken);
        } else {
            throw new AccountException(AccountExceptionEnum.PICA_LOGIN_AGAIN);
        }
    }

    @ApiOperation(value = "web登录获取用户信息")
    @GetMapping("/login/web")
    public PicaResponse<LoginResultWeb> getLoginResult() {
        LoginResultWeb resultWeb = new LoginResultWeb();
        long doctorId = super.getDoctorIdByToken();
        if (doctorId <= 0) {
            resultWeb.setIsExist(2);
            return PicaResponse.toResponse(resultWeb);
        }

        PICAPDoctor doctor = loginService.queryDoctor(doctorId);
        resultWeb.setPicapDoctor(doctor);
        if (doctor.getStatus() != null) {
            resultWeb.setCertifyDoc(doctor.getStatus().intValue());
        }
        return PicaResponse.toResponse(resultWeb);
    }

    @ApiOperation(value = "统一校验(传空则不会校验)")
    @PostMapping("/unifiedVerification")
    public PicaResponse unifiedVerification(@RequestBody EncryptEntity entity) throws Exception{
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        return PicaResponse.toResponse(loginService.preLoginValidate(request));
    }


    @ApiOperation(value = "app端手机号码一键登录")
    @PostMapping("/login/one-click")
    public PicaResponse<LoginResult> oneClickLogin(@RequestBody EncryptEntity entity) throws Exception {
        OneClickLoginReq req = CryptoUtil.decrypt(entity, OneClickLoginReq.class);
        req.setType(super.getSourceType());
        OneClickLoginResultVo oneClickLoginResultVo = loginService.oneClickLogin(req);
        return PicaResponse.toResponse(oneClickLoginResultVo);
    }
}
