package com.pica.cloud.account.account.server.service.impl;

import com.pica.cloud.account.account.server.entity.Account;
import com.pica.cloud.account.account.server.mapper.AccountMapper;
import com.pica.cloud.account.account.server.service.AccountService;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.EncryptCreateUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author andong
 * @create 2019/5/20
 */
@Service
public class AccountServiceImpl implements AccountService {

    private static final String KEY_PREFIX = "token-doctor-";
    private static Logger logger = LoggerFactory.getLogger(AccountServiceImpl.class);

    @Autowired
    private AccountMapper accountMapper;
    @Autowired
    @Qualifier("cacheMigrateClient")
    private ICacheClient cacheClient;

    //根据ID获取账号
    @Override
    public Account getById(long id) {
        return accountMapper.selectById(id);
    }

    //根据手机号获取账号
    @Override
    public Account getByMobilePhone(String mobilePhone) {
        String encryptMobilePhone = EncryptCreateUtil.encrypt(mobilePhone);
        return accountMapper.getByMobilePhone(encryptMobilePhone);
    }

    //根据微信unionid获取账号
    public Account getByUnionid(String unionid) {
        return accountMapper.getByUnionid(unionid);
    }

    //创建账号
    @Override
    @Transactional
    public void createAccount(Account account) {
        Date currentTime = new Date();
        account.setMobilePhone(EncryptCreateUtil.encrypt(account.getMobilePhone()));  //手机号加密
        account.setDeleteFlag(1);
        account.setCreatId(0L);
        account.setModifyId(0L);
        account.setCreatTime(currentTime);
        account.setModifyTime(currentTime);
        account.setFirstLoginTime(currentTime);
        account.setLastLoginTime(currentTime);
        accountMapper.insertSelective(account);
    }

    //更新账号信息
    @Override
    @Transactional
    public void updateAccountById(Account account) {
        accountMapper.updateByIdSelective(account);
    }

    @Override
    public Map<String, String> getCache(long id) {
        String key = KEY_PREFIX + id;
        Map<String, String> map = new HashMap();
        map.put("id", cacheClient.hget(key, "id"));
        map.put("hospital_id", cacheClient.hget(key, "hospital_id"));
        map.put("hospital", cacheClient.hget(key, "hospital"));
        return map;
    }

    @Override
    public void refreshCache(long id) {
        int offset = 0;
        final int size = 2000;

        if (id >= 0) {  //刷新单个医生缓存信息
            List<Map<String, Object>> list = accountMapper.getHospitalInfoByPage(id, offset, size);
            if (CollectionUtils.isNotEmpty(list)) {
                Map<String, Object> doctorMap = list.get(0);
                this.refresh(doctorMap);
            }
        } else {  //刷新全部医生缓存信息
            int errorCount = 0;  //异常次数
            List<Map<String, Object>> list = accountMapper.getHospitalInfoByPage(null, offset, size);
            while (CollectionUtils.isNotEmpty(list)) {
                for (Map<String, Object> doctorMap : list) {
                    int result = this.refresh(doctorMap);
                    errorCount += result;
                }

                if (errorCount >= 10000) {
                    break;  //错误次数超过一定数量停止任务
                }

                logger.info("更新用户缓存信息已完成数量：{}", offset + list.size());
                offset += size;
                list = accountMapper.getHospitalInfoByPage(null, offset, size);
            }

            logger.info("更新用户缓存信息全部完成！");
        }
    }

    //更新缓存中hospital_id，hospital信息
    //成功返回0，失败返回1
    private int refresh(Map<String, Object> doctorMap) {
        String doctorId = doctorMap.get("id").toString();
        String hospitalId = doctorMap.get("hospital_id") == null ? StringUtils.EMPTY : doctorMap.get("hospital_id").toString();
        String hospital = doctorMap.get("hospital") == null ? StringUtils.EMPTY : doctorMap.get("hospital").toString();

        String key = KEY_PREFIX + doctorId;
        try {
            String cacheId = cacheClient.hget(key, "id");
            if (StringUtils.isBlank(cacheId)) {
                return 0;  //未找到用户缓存数据，不做更新
            }
            cacheClient.hset(key, "hospital_id", StringUtils.isEmpty(hospitalId) ? "0" : hospitalId);
            cacheClient.hset(key, "hospital", hospital);
            return 0;
        } catch (Exception ex) {
            return 1;
        }
    }
}
