// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.pica.cloud.account.account.common.req.AcctPatFamilyDto;
import com.pica.cloud.account.account.common.req.AcctPatFamilyReq;
import com.pica.cloud.account.account.common.req.AcctPatInfoReq;
import com.pica.cloud.account.account.common.resp.PatBindResp;
import com.pica.cloud.account.account.common.resp.PatFamilyResp;
import com.pica.cloud.account.account.server.entity.AcctPatFamily;
import com.pica.cloud.account.account.server.entity.AcctPatInfo;
import com.pica.cloud.account.account.server.entity.AcctPatUnion;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.mapper.AcctPatFamilyMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatInfoMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatUnionMapper;
import com.pica.cloud.account.account.server.service.PatHealthPackService;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResponse;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static java.util.stream.Collectors.toList;

/**
 * @ClassName PatHealthPackServiceIml
 * @Description TODO
 * @Author peijun.zhao
 * @Date 2020/5/14 18:30
 * @ModifyDate 2020/5/14 18:30
 * @Version 1.0
 */
@Service
public class PatHealthPackServiceIml implements PatHealthPackService {

    @Autowired
    private AcctPatInfoMapper patInfoMapper;

    @Autowired
    private AcctPatUnionMapper patUnionMapper;

    @Autowired
    private AcctPatFamilyMapper patFamilyMapper;

    @Override
    public Integer getAcctIdByUnionId(String unionId) {
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(unionId);
        return patUnion == null ? 0 : patUnion.getAcctId();
    }

    /**
     * @Description 获取居民id是本人的家庭组记录-
     * 作为家庭主账户、本人居民id，单条ok 多条异常数据
     * @Author peijun.zhao
     * @Date 2020/5/14 18:48
     * @ModifyDate 2020/5/14 18:48
     * @Params [patientId]
     * @Return java.lang.Integer
     */
    @Override
    public Integer getAcctIdByPatId(Integer patientId) {
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getByPatIdSelf(patientId);
        if (CollectionUtils.isEmpty(patFamilyList)) {
            //该居民id无 家庭成员记录，返回0
            return 0;
        }
        if (patFamilyList.size() > 1) {
            //db中数据有异常,log  异常
            throw new PicaException(AccountExceptionEnum.PAT_ACCT_HAS_MORE.getCode(), AccountExceptionEnum.PAT_ACCT_HAS_MORE.getMessage());
        }
        return patFamilyList.get(0).getAcctId();
    }

    @Override
    public List<Integer> getDirectPatIdsByPatId(Integer patientId) {
        List<Integer> patIds = new ArrayList<>();
        List<AcctPatFamily> memberList = patFamilyMapper.getByPatIdNotSelf(patientId);
        if (CollectionUtils.isEmpty(memberList)) {
            //该居民id无 家庭成员记录，返回null
            return patIds;
        }
        if (memberList.size() > 1) {
            //db中数据有异常,log 抛异常
            throw new PicaException(AccountExceptionEnum.PAT_MEMBER_HAS_MORE.getCode(), AccountExceptionEnum.PAT_MEMBER_HAS_MORE.getMessage());
        }

        //add 病人id所在家庭组的监护人- 上级
        AcctPatFamily parentMember = patFamilyMapper.getSelfByAcctId(memberList.get(0).getAcctId());
        patIds.add(parentMember.getPatientId());

        //下级所有成员 包括自身
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(memberList.get(0).getAcctId());
        patIds.addAll(patFamilyList.stream().map(obj -> obj.getPatientId()).collect(toList()));
        return patIds;
    }


    @Override
    public PatFamilyResp getFamilyPatsByAcctId(Integer acctId) {
        PatFamilyResp resp = new PatFamilyResp();
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(acctId);
        resp.setMemberPatIds(patFamilyList.stream().map(obj ->obj.getPatientId()).collect(toList()));
        List<AcctPatFamilyDto> dtos =  new ArrayList<>();
        for(AcctPatFamily family : patFamilyList){
            AcctPatFamilyDto dto = new AcctPatFamilyDto();
            dto.setPatientId(family.getPatientId());
            dto.setRelation(family.getRelation());
            dtos.add(dto);
        }
        resp.setFailMemberList(dtos);
        return resp;
    }

    @Transactional
    @Override
    public Integer saveAcct(AcctPatInfoReq patInfoReq) {
        if(Objects.isNull(patInfoReq.getPatientId()) || patInfoReq.getPatientId() == 0 || Objects.isNull(patInfoReq.getName())
                || Objects.isNull(patInfoReq.getMobile()) ){
            throw new PicaException(AccountExceptionEnum.PAT_SAVE_PARAM_ERROR.getCode(), AccountExceptionEnum.PAT_SAVE_PARAM_ERROR.getMessage());
        }

        String mobileEncrypt = EncryptUtils.encryptContent(patInfoReq.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE);
        //check 手机号是否已是主账户
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getByPatIdSelf(patInfoReq.getPatientId());
        if (CollectionUtils.isNotEmpty(patFamilyList)) {
            // 重复时返回已有账户
            return patFamilyList.get(0).getAcctId();
        }

        //手机号无重复时创建
        AcctPatInfo patInfo = new AcctPatInfo();
        patInfo.setName(patInfoReq.getName());
        patInfo.setMobilePhone(mobileEncrypt);
        patInfo.setRegisterProduct(1);
        patInfo.setRegisterSource(1);
        patInfo.setRegTime(new Date());

        patInfo.setDeleteFlag(1);
        patInfo.setCreatedId(0);
        patInfo.setModifiedId(0);
        patInfo.setCreatedTime(new Date());
        patInfo.setModifiedTime(new Date());
        patInfoMapper.insert(patInfo);

        AcctPatFamily patFamily = new AcctPatFamily();
        patFamily.setAcctId(patInfo.getId());
        patFamily.setPatientId(patInfoReq.getPatientId());
        patFamily.setRelation(1);

        patFamily.setDeleteFlag(1);
        patFamily.setCreatedId(0);
        patFamily.setModifiedId(0);
        patFamily.setCreatedTime(new Date());
        patFamily.setModifiedTime(new Date());

        patFamilyMapper.insert(patFamily);
        return patInfo.getId();
    }

    @Transactional
    @Override
    public PatBindResp bindUnion(AcctPatInfoReq patInfoReq) {
        PatBindResp resp = new PatBindResp();
        //unionid是否已被其他账号绑定
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(patInfoReq.getUnionId());
        if (Objects.nonNull(patUnion)) {
            //被绑定
            //且非传入acct_d
            if (!patUnion.getAcctId().equals(patInfoReq.getAcctId())) {
                resp.setBindStatus(false);
                resp.setMsg(AccountExceptionEnum.PAT_UNIONID_BINDED_ERROR.getMessage());
            } else {
                //unionId,已被传入acctId绑定，返回 true
                resp.setBindStatus(true);
            }
        } else {
            //acctId是否已绑定其他unionId
            patUnion = patUnionMapper.selectByAcctId(patInfoReq.getAcctId());
            if (Objects.nonNull(patUnion)) {
                //acctId已有unionId，且不是传入的unionId
                if (!patUnion.getUnionId().equals(patInfoReq.getUnionId())) {
                    resp.setBindStatus(false);
                    resp.setMsg(AccountExceptionEnum.PAT_MOBILE_BINDED_WECHAT_ERROR.getMessage());
                }
            } else {
                //未绑定 新增绑定记录
                AcctPatUnion entity = new AcctPatUnion();
                entity.setAcctId(patInfoReq.getAcctId());
                entity.setUnionId(patInfoReq.getUnionId());
                entity.setUnionType(1);

                entity.setDeleteFlag(1);
                entity.setCreatedId(patInfoReq.getAcctId());
                entity.setModifiedId(patInfoReq.getAcctId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());

                patUnionMapper.insert(entity);
                resp.setBindStatus(true);
            }
        }

        return resp;
    }

    @Transactional
    @Override
    public PatFamilyResp saveMember(AcctPatFamilyReq familyReqReq) {
        PatFamilyResp resp = new PatFamilyResp();
        boolean successFlag = true;

        //check member valid
        List<AcctPatFamilyDto> memberList = familyReqReq.getMemberList();
        List<AcctPatFamilyDto> failList = new ArrayList<>();
        List<Integer> patIds = memberList.stream().map(obj -> obj.getPatientId()).collect(toList());
        List<AcctPatFamilyDto> relatedPats = new ArrayList<>();
        List<AcctPatFamily> familyList = patFamilyMapper.getListByPatIds(patIds);
        for(AcctPatFamily member : familyList){
            if(!familyReqReq.getAcctId().equals(member.getAcctId())){
                //居民已被其他账户绑定为家庭成员
                AcctPatFamilyDto failOne = new AcctPatFamilyDto();
                failOne.setPatientId(member.getPatientId());
                failOne.setRelation(member.getRelation());
                failList.add(failOne);
                successFlag = false;
            }else {
                //居民已被传入acctId关联为家庭成员，不需要再保存
                AcctPatFamilyDto failOneRelated = new AcctPatFamilyDto();
                failOneRelated.setRelation(member.getRelation());
                failOneRelated.setPatientId(member.getPatientId());
                relatedPats.add(failOneRelated);
                successFlag = false;
            }
        }

        //已关联自身居民list
        if(relatedPats.size() > 0){
            successFlag = false;
            failList.addAll(relatedPats);
        }

        //通过后 再建立家庭关系
        if(successFlag){
            for(AcctPatFamilyDto member : memberList){
                AcctPatFamily entity = new AcctPatFamily();
                entity.setAcctId(familyReqReq.getAcctId());
                entity.setPatientId(member.getPatientId());
                entity.setRelation(member.getRelation());

                entity.setDeleteFlag(1);
                entity.setCreatedId(familyReqReq.getAcctId());
                entity.setModifiedId(familyReqReq.getAcctId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());
                patFamilyMapper.insert(entity);
            }
        }

        resp.setSuccessFlag(successFlag);
        resp.setFailMemberList(failList);
        return resp;
    }

    @Override
    public Integer getFamilyAcctPatId(Integer patId) {
        Integer rtnPatId = 0;
        List<AcctPatFamily> acctMembers = patFamilyMapper.getByPatIdSelf(patId);
        if(CollectionUtils.isNotEmpty(acctMembers)){
            //本人为监护人 返回自身patId
            rtnPatId = acctMembers.get(0).getPatientId();
        }else {
            //查询是否存在 作为成员的记录
            List<AcctPatFamily> members = patFamilyMapper.getByPatIdNotSelf(patId);
            if(CollectionUtils.isNotEmpty(members)){
                //本人为成员时记录存在
                AcctPatFamily acctPat = patFamilyMapper.getSelfByAcctId(members.get(0).getAcctId());
                rtnPatId = acctPat.getPatientId();
            }else {
                //传入patId不存在家庭记录
            }
        }
        return rtnPatId;
    }
}
