package com.pica.cloud.account.account.server.service.impl;

import com.pica.cloud.account.account.server.entity.Account;
import com.pica.cloud.account.account.server.entity.AccountInfoEntity;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.mapper.AccountInfoDetailMapper;
import com.pica.cloud.account.account.server.mapper.AccountMapper;
import com.pica.cloud.account.account.server.service.AccountService;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @author andong
 * @create 2019/5/20
 */
@Service
public class AccountServiceImpl implements AccountService {

    private static final String KEY_PREFIX = "token-doctor-";
    private static Logger logger = LoggerFactory.getLogger(AccountServiceImpl.class);

    @Autowired
    private AccountMapper accountMapper;

    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;
    @Autowired
    @Qualifier("cacheMigrateClient")
    private ICacheClient cacheClient;

    @Value("${doubleWritingMode}")
    private boolean doubleWritingMode;

    //根据ID获取账号
    @Override
    public Account getById(long id) {
        return accountMapper.selectById(id);
    }

    //根据手机号获取账号
    @Override
    public Account getByMobilePhone(String mobilePhone) {
        String encryptMobilePhone = EncryptUtils.encryptContent(mobilePhone, EncryptConstants.ENCRYPT_TYPE_MOBILE);
        return accountMapper.getByMobilePhone(encryptMobilePhone);
    }

    @Override
    public AccountInfoEntity getAccountByMobilePhone(String mobilePhone) {
        String encryptMobilePhone = EncryptUtils.encryptContent(mobilePhone, EncryptConstants.ENCRYPT_TYPE_MOBILE);
        return accountInfoDetailMapper.selectByMobile(encryptMobilePhone);
    }

    @Override
    public Integer getAccountIdByMobilePhone(String mobilePhone) {
        String encryptMobilePhone = EncryptUtils.encryptContent(mobilePhone, EncryptConstants.ENCRYPT_TYPE_MOBILE);
        return accountInfoDetailMapper.selectAcctIdByMobile(encryptMobilePhone);
    }

    //根据微信unionid获取账号
    public Account getByUnionid(String unionid) {
        return accountMapper.getByUnionid(unionid);
    }

    //创建账号
    @Override
    @Transactional
    public void createAccount(Account account) {
        //账户表
        AccountInfoEntity accountInfo = new AccountInfoEntity();
        Date currentTime = new Date();
        accountInfo.setMobilePhone(account.getMobilePhone());
        accountInfo.setPassword("");
        accountInfo.setCreatedTime(currentTime);
        accountInfo.setCreatedId(0);
        accountInfo.setModifiedId(0);
        accountInfo.setModifiedTime(currentTime);
        accountInfo.setRegTime(currentTime);
        accountInfo.setDeleteFlag(1);
        accountInfo.setSex(0);
        accountInfo.setRegisterProduct(AccountTypeEnum.PRODUCT_TYPE_DOCTOR.getCode());
        accountInfo.setRegisterSource(AccountTypeEnum.DEVICE_TYPE_H5.getCode());
        accountInfoDetailMapper.insertSelective(accountInfo);
        Integer acctId = accountInfo.getId();
        //doctor表,存入用户id
        if (doubleWritingMode) {
            account.setMobilePhone(account.getMobilePhone());
            account.setDeleteFlag(1);
            account.setCreatId(0L);
            account.setModifyId(0L);
            account.setAcctId(acctId);
            account.setCreatTime(currentTime);
            account.setRegisterSource(AccountTypeEnum.DEVICE_TYPE_H5.getCode());
            account.setModifyTime(currentTime);
            account.setFirstLoginTime(currentTime);
            account.setLastLoginTime(currentTime);
            accountMapper.insertSelective(account);
        }
    }

    //更新账号信息
    @Override
    @Transactional
    public void updateAccountById(Account account) {
        accountMapper.updateByIdSelective(account);
    }

    @Override
    public Map<String, String> getCache(long id) {
        String key = KEY_PREFIX + id;
        Map<String, String> map = new HashMap();
        map.put("id", cacheClient.hget(key, "id"));
        map.put("hospital_id", cacheClient.hget(key, "hospital_id"));
        map.put("hospital", cacheClient.hget(key, "hospital"));
        map.put("name", cacheClient.hget(key, "name"));
//        Map<String, String> result = cacheClient.hgetall(key);
//        Iterator<Map.Entry<String, String>> iterator = result.entrySet().iterator();
//        while(iterator.hasNext()){
//            Map.Entry<String, String> next = iterator.next();
//            String key1 = next.getKey();
//            String value = next.getValue();
//            logger.info("更新用户缓存信息已完成数量：{}", key1 + "------>"+value);
//        }
//        PicaUser picaUser = cacheClient.getToken("F64279B337EA4AE3965853A7CCFDF5A3", PicaUser.class);
//        logger.info("更新用户缓存信息已完成数量：", picaUser.toString());
        return map;
    }

    @Override
    public void refreshCache(long id) {
        int offset = 0;
        final int size = 5000;

        if (id >= 0) {  //刷新单个医生缓存信息
            List<Map<String, Object>> list = accountMapper.getHospitalInfoByPage(id, offset, size);
            if (CollectionUtils.isNotEmpty(list)) {
                Map<String, Object> doctorMap = list.get(0);
                this.refresh(doctorMap);
            }
        } else {  //刷新全部医生缓存信息
            int errorCount = 0;  //异常次数
            List<Map<String, Object>> list = accountMapper.getHospitalInfoByPage(null, offset, size);
            while (CollectionUtils.isNotEmpty(list)) {
                for (Map<String, Object> doctorMap : list) {
                    int result = this.refresh(doctorMap);
                    errorCount += result;
                }

                if (errorCount >= 10000) {
                    break;  //错误次数超过一定数量停止任务
                }

                logger.info("更新用户缓存信息已完成数量：{}", offset + list.size());
                offset += size;
                list = accountMapper.getHospitalInfoByPage(null, offset, size);
            }

            logger.info("更新用户缓存信息全部完成！");
        }
    }


    //更新缓存中hospital_id，hospital信息
    //成功返回0，失败返回1
    private int refresh(Map<String, Object> doctorMap) {
        String doctorId = doctorMap.get("id").toString();
        /*
        String hospitalId = doctorMap.get("hospital_id") == null ? StringUtils.EMPTY : doctorMap.get("hospital_id").toString();
        String hospital = doctorMap.get("hospital") == null ? StringUtils.EMPTY : doctorMap.get("hospital").toString();
        String mobile = doctorMap.get("mobile_phone") == null ? StringUtils.EMPTY : doctorMap.get("mobile_phone").toString();
        String status = doctorMap.get("status") == null ? StringUtils.EMPTY : doctorMap.get("status").toString();
        String avatar_image_url = doctorMap.get("avatar_image_url") == null ? StringUtils.EMPTY : doctorMap.get("avatar_image_url").toString();
        String email = doctorMap.get("email") == null ? StringUtils.EMPTY : doctorMap.get("email").toString();
        String province = doctorMap.get("province") == null ? StringUtils.EMPTY : doctorMap.get("province").toString();
        String province_name = doctorMap.get("province") == null ? StringUtils.EMPTY : doctorMap.get("province_name").toString();
        String city = doctorMap.get("city") == null ? StringUtils.EMPTY : doctorMap.get("city").toString();
        String city_name = doctorMap.get("city_name") == null ? StringUtils.EMPTY : doctorMap.get("city_name").toString();
        String county = doctorMap.get("county") == null ? StringUtils.EMPTY : doctorMap.get("county").toString();
        String county_name = doctorMap.get("county_name") == null ? StringUtils.EMPTY : doctorMap.get("county_name").toString();
        String town = doctorMap.get("town") == null ? StringUtils.EMPTY : doctorMap.get("town").toString();
        String town_name = doctorMap.get("town_name") == null ? StringUtils.EMPTY : doctorMap.get("town_name").toString();
        String title = doctorMap.get("title") == null ? StringUtils.EMPTY : doctorMap.get("title").toString();
        String title_id = doctorMap.get("title_id") == null ? StringUtils.EMPTY : doctorMap.get("title_id").toString();
         */

        String name = doctorMap.get("name") == null ? StringUtils.EMPTY : doctorMap.get("name").toString();
        String key = KEY_PREFIX + doctorId;
        try {
            String cacheId = cacheClient.hget(key, "id");
            if (StringUtils.isBlank(cacheId)) {
                return 0;  //未找到用户缓存数据，不做更新
            }
            /**
             cacheClient.hset(key, "hospital_id", StringUtils.isEmpty(hospitalId) ? "0" : hospitalId);
             cacheClient.hset(key, "hospital", hospital);
             cacheClient.hset(key, "province_name", province_name);
             cacheClient.hset(key, "city_name", city_name);
             cacheClient.hset(key, "county_name", county_name);
             cacheClient.hset(key, "town_name", town_name);
             cacheClient.hset(key, "province_id", province);
             cacheClient.hset(key, "city_id", city);
             cacheClient.hset(key, "county_id", county);
             cacheClient.hset(key, "town_id", town);
             cacheClient.hset(key, "mobile", mobile);
             cacheClient.hset(key, "status", status);
             cacheClient.hset(key, "avatar_image_url", avatar_image_url);
             cacheClient.hset(key, "email", email);
             cacheClient.hset(key, "title_id", title_id);
             cacheClient.hset(key, "title", title);
             */

            cacheClient.hset(key, "name", name);
            return 0;
        } catch (Exception ex) {
            return 1;
        }
    }
}
