// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.RcRepTypeEnum;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.model.JigsawCodeModel;
import com.pica.cloud.account.account.server.service.AuthCodeService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.IPUtil;
import com.pica.cloud.account.account.server.util.PICAPDeviceInfo;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.entity.PicaWarnException;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.StringUtil;
import com.pica.cloud.proof.client.SliderImageService;
import com.pica.cloud.proof.contract.request.SliderImageRequest;
import com.pica.cloud.riskcontrol.riskcontrol.client.CheckCodeClient;
import com.pica.cloud.riskcontrol.riskcontrol.common.req.CheckcodeRiskReq;
import com.pica.cloud.riskcontrol.riskcontrol.common.resp.CheckcodeRiskResp;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * @ClassName AuthCodeServiceImpl
 * @Description
 * @Author Chongwen.jiang
 * @Date 2020/3/5 17:01
 * @ModifyDate 2020/3/5 17:01
 * @Version 1.0
 */
@Service
public class AuthCodeServiceImpl implements AuthCodeService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private CheckCodeClient rcClient;

    @Autowired
    private ICacheClient cacheClient;

    @Autowired
    @Qualifier("accountThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private SliderImageService sliderImageService;


    /**
     * @Description 调用风控接口
     * @Author Chongwen.jiang
     * @Date 2020/2/21 17:12
     * @ModifyDate 2020/2/21 17:12
     * @Params [mobile]
     * @Return com.pica.cloud.account.account.server.resp.UnifiedVerificationResp
     */
    @Override
    public void rcValidate(HttpServletRequest request, BaseRequest baseRequest, String deviceInfoStr) {
        //  调用风控接口
        PicaResponse picaResponse = null;
        try {
            PICAPDeviceInfo deviceInfo = JSON.parseObject(deviceInfoStr, PICAPDeviceInfo.class);
            String publicIp = IPUtil.getIpAdrress(request);
            String deviceToken = baseRequest.getDevice_token();
            CheckcodeRiskReq rcReq = new CheckcodeRiskReq();
            rcReq.setMobile(baseRequest.getMobile());
            rcReq.setIp(publicIp);
            rcReq.setDeviceIp(deviceInfo.getDevice_ip());
            rcReq.setDeviceId(deviceToken);
            logger.info("rc-checkcodeRisk-req:{}", JSON.toJSONString(rcReq));
            picaResponse = rcClient.checkcodeRisk(rcReq);
            logger.info("rc-checkcodeRisk-resp:{}", JSON.toJSONString(picaResponse));
        } catch (JSONException e) {
            logger.error("rc-checkcodeRisk-JSONException", e);
        } catch (Exception e) {
            logger.error("rc-checkcodeRisk-invoke-exception", e);
        }

        if (picaResponse != null &&
                PicaResultCode.SUCCESS.code().equals(picaResponse.getCode())) {
            Object data = picaResponse.getData();
            if (Objects.nonNull(data)) {
                CheckcodeRiskResp respData = JSON.parseObject(
                        JSON.toJSONString(data), CheckcodeRiskResp.class);
                if (StringUtils.isNotEmpty(respData.getProcessCode())) {
                    String processCode = respData.getProcessCode();
                    if (StringUtils.isNotEmpty(processCode)) {
                        //  请求返回正常
                        if (processCode.equals(RcRepTypeEnum.RC_SEND_FAIL.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_SEND_FAIL.getCode(),
                                    AccountExceptionEnum.PICA_RC_SEND_FAIL.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_TRIGGER.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_TRIGGER.getCode(),
                                    AccountExceptionEnum.PICA_RC_TRIGGER.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_BLACKLIST.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_BLACKLIST.getCode(),
                                    AccountExceptionEnum.PICA_RC_BLACKLIST.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_DAY_LIMIT.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_DAY_LIMIT.getCode(),
                                    AccountExceptionEnum.PICA_RC_DAY_LIMIT.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_60_SECOND_LIMIT.getCode())) {
                            //  获取剩余秒数
                            String authCodeKey = Constants.AUTH_CODE_PREFIX + baseRequest.getFlag() + "-" + AESUtil.encryptV0(baseRequest.getMobile()) + "-secure";
                            Long time = cacheClient.get(authCodeKey, Long.class);
                            if (Objects.nonNull(time)) {
                                int remainTime = 59 - (int) (System.currentTimeMillis() - time) / 1000;
                                if (remainTime > 0) {
                                    throw new PicaWarnException(AccountExceptionEnum.PICA_RC_60_SECOND_LIMIT.getCode(),
                                            AccountExceptionEnum.PICA_RC_60_SECOND_LIMIT.getMessage().replace("{remainTime}", String.valueOf(remainTime)));
                                }
                            } else {
                                //  刚好60秒缓存时间过去了，也允许发送短信验证码
                            }
                        } else {
                            //  processCode equals "1" (允许发送短信验证码)
                        }
                    }
                }
            }
        }

    }

    /**
     * @Description 记录风控数据
     * @Author Chongwen.jiang
     * @Date 2020/2/27 16:53
     * @ModifyDate 2020/2/27 16:53
     * @Params [req, request]
     * @Return void
     */
    @Override
    public void recordRcData(HttpServletRequest request, BaseRequest baseRequest, String deviceIp) {
        CheckcodeRiskReq rec = new CheckcodeRiskReq();
        rec.setMobile(baseRequest.getMobile());
        rec.setIp(IPUtil.getIpAdrress(request));
        rec.setDeviceIp(deviceIp);
        rec.setDeviceId(baseRequest.getDevice_token());
        try {
            threadPoolTaskExecutor.execute(() -> {
                logger.info("recordRcData-execute-{}", JSON.toJSONString(rec));
                PicaResponse resp = rcClient.logSend(rec);
                logger.info("recordRcData-resp-{}", JSON.toJSONString(resp));
            });
        } catch (Exception e) {
            logger.error("recordRcData invoke error", e.getMessage());
        }
    }

    /**
     * @Description 拼图验证
     * @Author Chongwen.jiang
     * @Date 2020/3/23 15:04
     * @ModifyDate 2020/3/23 15:04
     * @Params [capCode, xPos]
     * @Return void
     */
    @Override
    public void validateJigsawCode(String mobile, String capCode, int xPos) {
        // 调用风控服务，是否需要触发拼图验证
        Boolean need = this.judgeNeedJigsaw(mobile);
        Boolean paramValid = StringUtils.isNotEmpty(capCode) && xPos > 0;
        if (!need && !paramValid) {
            return;
        }

        if (!paramValid) {
            throw new PicaWarnException(PicaResultCode.PARAM_IS_INVALID.code(), PicaResultCode.PARAM_IS_INVALID.message());
        }
        PicaResponse resp = null;
        try {
            SliderImageRequest req = new SliderImageRequest();
            req.setCapcode(capCode);
            req.setXpos(xPos);
            resp = sliderImageService.checkcapcode(req);
            logger.info("validateJigsawCode-resp:{}", JSON.toJSONString(resp));
        } catch (Exception e) {
            logger.error("validateJigsawCode-sliderImageService.checkcapcode invoke exception", e.getMessage());
            throw new PicaWarnException(AccountExceptionEnum.PICA_CAP_CODE_ERROR.getCode(),
                    AccountExceptionEnum.PICA_CAP_CODE_ERROR.getMessage());
        }

        String code = "2";
        if (Objects.nonNull(resp)) {
            JSONObject respData = JSON.parseObject(JSON.toJSONString(resp.getData()), JSONObject.class);
            if (Objects.nonNull(respData)) {
                Object codeObj = respData.get("code");
                if (Objects.nonNull(codeObj)) {
                    code = String.valueOf(codeObj);
                }
            }
        }
        if (StringUtil.equals(code, "2") ||
                StringUtil.equals(code, "3")) {
            // 3超期 2验证失败 1成功
            throw new PicaWarnException(AccountExceptionEnum.PICA_CAP_CODE_ERROR.getCode(),
                    AccountExceptionEnum.PICA_CAP_CODE_ERROR.getMessage());
        }
    }

    /**
     * @Description 获取拼图码校验
     * @Author Chongwen.jiang
     * @Date 2020/3/23 15:29
     * @ModifyDate 2020/3/23 15:29
     * @Params [mobile]
     * @Return com.pica.cloud.account.account.server.model.JigsawCodeModel
     */
    @Override
    public JigsawCodeModel validateJigsawCode(String mobile) {
        logger.info("getJigsawCode-mobile:{}", mobile);
        // 调用风控服务，是否需要触发拼图验证
        Boolean need = this.judgeNeedJigsaw(mobile);

        //  调用proof服务获取拼图数据
        JigsawCodeModel data = new JigsawCodeModel();
        PicaResponse picResp = null;
        if (need) {
            try {
                picResp = sliderImageService.getPic();
                logger.info("getJigsawCode-getPic:{}", JSON.toJSONString(picResp));
            } catch (Exception e) {
                logger.error("getJigsawCode-sliderImageService.getPic invoke exception", e.getMessage());
                throw new PicaWarnException(AccountExceptionEnum.PICA_CAP_GET_INVOKE_ERROR.getCode(),
                        AccountExceptionEnum.PICA_CAP_GET_INVOKE_ERROR.getMessage());
            }
            if (Objects.nonNull(picResp)) {
                JSONObject respData = JSON.parseObject(JSON.toJSONString(picResp.getData()), JSONObject.class);
                if (Objects.nonNull(respData)) {
                    data.setCapCode(respData.getString("capcode"));
                    data.setBackImage(respData.getString("backImage"));
                    data.setSlidingImage(respData.getString("slidingImage"));
                    data.setyHeight(respData.getInteger("yHeight"));
                }
            }
        }

        return data;
    }

    /**
     * @Description 调用风控服务，是否需要触发拼图验证
     * @Author Chongwen.jiang
     * @Date 2020/3/23 16:01
     * @ModifyDate 2020/3/23 16:01
     * @Params [mobile]
     * @Return java.lang.Boolean true:需要触发拼图验证 false:不需要
     */
    private Boolean judgeNeedJigsaw(String mobile) {
        Boolean need = true;
        try {
            //  TODO 调用风控服务，是否需要触发拼图验证
            //  rcClient.

        } catch (Exception e) {
            logger.error("judgeNeedJigsaw-client invoke exception", e.getMessage());
            throw new PicaWarnException(PicaResultCode.INTERFACE_INVOKE_ERROR.code(),
                    PicaResultCode.INTERFACE_INVOKE_ERROR.message());
        }
        return need;
    }

}
