// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.RcRepTypeEnum;
import com.pica.cloud.account.account.server.enums.SourceTypeEnum;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.model.JigsawCodeModel;
import com.pica.cloud.account.account.server.service.AuthCodeService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.AccountUtils;
import com.pica.cloud.account.account.server.util.IPUtil;
import com.pica.cloud.account.account.server.util.PICAPDeviceInfo;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.entity.PicaWarnException;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.StringUtil;
import com.pica.cloud.proof.client.SliderImageService;
import com.pica.cloud.proof.contract.request.SliderImageRequest;
import com.pica.cloud.riskcontrol.riskcontrol.client.CheckCodeClient;
import com.pica.cloud.riskcontrol.riskcontrol.common.req.CheckcodeRiskReq;
import com.pica.cloud.riskcontrol.riskcontrol.common.resp.CheckcodeRiskResp;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * @ClassName AuthCodeServiceImpl
 * @Description
 * @Author Chongwen.jiang
 * @Date 2020/3/5 17:01
 * @ModifyDate 2020/3/5 17:01
 * @Version 1.0
 */
@Service
public class AuthCodeServiceImpl implements AuthCodeService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private CheckCodeClient rcClient;

    @Autowired
    private ICacheClient cacheClient;

    @Autowired
    @Qualifier("accountThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private SliderImageService sliderImageService;


    /**
     * @Description 调用风控接口
     * @Author Chongwen.jiang
     * @Date 2020/2/21 17:12
     * @ModifyDate 2020/2/21 17:12
     * @Params [mobile]
     * @Return com.pica.cloud.account.account.server.resp.UnifiedVerificationResp
     */
    @Override
    public int rcValidate(HttpServletRequest request, BaseRequest baseRequest, String deviceInfoStr, Integer sourceType) {
        int needCheck = 1;
        //  调用风控接口
        PicaResponse picaResponse = null;
        try {
            PICAPDeviceInfo deviceInfo = JSON.parseObject(deviceInfoStr, PICAPDeviceInfo.class);
            String publicIp = IPUtil.getIpAdrress(request);
            String deviceToken = baseRequest.getDevice_token();
            CheckcodeRiskReq rcReq = new CheckcodeRiskReq();
            rcReq.setMobile(baseRequest.getMobile());
            rcReq.setIp(publicIp);
            rcReq.setDeviceIp(deviceInfo.getDevice_ip());
            rcReq.setDeviceId(deviceToken);
            rcReq.setSourceType(sourceType);
            logger.info("rc-checkcodeRisk-req:{}", JSON.toJSONString(rcReq));
            picaResponse = rcClient.checkcodeRisk(rcReq);
            logger.info("rc-checkcodeRisk-{}-resp:{}", baseRequest.getMobile(), JSON.toJSONString(picaResponse));
        } catch (JSONException e) {
            logger.error("rc-checkcodeRisk-JSONException", e);
        } catch (Exception e) {
            logger.error("rc-checkcodeRisk-invoke-exception", e);
        }

        if (picaResponse != null &&
                PicaResultCode.SUCCESS.code().equals(picaResponse.getCode())) {
            Object data = picaResponse.getData();
            if (Objects.nonNull(data)) {
                CheckcodeRiskResp respData = JSON.parseObject(
                        JSON.toJSONString(data), CheckcodeRiskResp.class);
                if (StringUtils.isNotEmpty(respData.getProcessCode())) {
                    String processCode = respData.getProcessCode();
                    if (StringUtils.isNotEmpty(processCode)) {
                        //  请求返回正常
                        if (processCode.equals(RcRepTypeEnum.RC_SEND_FAIL.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_SEND_FAIL.getCode(),
                                    AccountExceptionEnum.PICA_RC_SEND_FAIL.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_TRIGGER.getCode())) {
                            if(SourceTypeEnum.H5.getCode().equals(sourceType)) {
                                // h5端需要触发拼图验证，后台去获取拼图数据返给前端
                            } else {
                                throw new PicaWarnException(AccountExceptionEnum.PICA_RC_TRIGGER.getCode(),
                                        AccountExceptionEnum.PICA_RC_TRIGGER.getMessage());
                            }
                        } else if (processCode.equals(RcRepTypeEnum.RC_BLACKLIST.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_BLACKLIST.getCode(),
                                    AccountExceptionEnum.PICA_RC_BLACKLIST.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_DAY_LIMIT.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_DAY_LIMIT.getCode(),
                                    AccountExceptionEnum.PICA_RC_DAY_LIMIT.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_60_SECOND_LIMIT.getCode())) {
                            //  获取剩余秒数
                            String authCodeKey = Constants.AUTH_CODE_PREFIX + baseRequest.getFlag() + "-" + AESUtil.encryptV0(baseRequest.getMobile()) + "-secure";
                            Long time = cacheClient.get(authCodeKey, Long.class);
                            if (Objects.nonNull(time)) {
                                int remainTime = 59 - (int) (System.currentTimeMillis() - time) / 1000;
                                if (remainTime > 0) {
                                    throw new PicaWarnException(AccountExceptionEnum.PICA_RC_60_SECOND_LIMIT.getCode(),
                                            AccountExceptionEnum.PICA_RC_60_SECOND_LIMIT.getMessage().replace("{remainTime}", String.valueOf(remainTime)));
                                } else {
                                    needCheck = 0;
                                }
                            } else {
                                //  刚好60秒缓存时间过去了，也允许发送短信验证码
                                needCheck = 0;
                            }
                        } else if (RcRepTypeEnum.RC_ALLOW_SEND_MSG.getCode().equals(processCode)) {
                            //  processCode equals "1" (允许发送短信验证码)
                            needCheck = -1;
                        } else {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_BUSINESS_ERROR.getCode(),
                                    AccountExceptionEnum.PICA_RC_BUSINESS_ERROR.getMessage());
                        }
                    }
                }
            }
        }

        return needCheck;
    }

    /**
     * @Description 记录风控数据
     * @Author Chongwen.jiang
     * @Date 2020/2/27 16:53
     * @ModifyDate 2020/2/27 16:53
     * @Params [req, request]
     * @Return void
     */
    @Override
    public void recordRcData(HttpServletRequest request, BaseRequest baseRequest, String deviceIp) {
        CheckcodeRiskReq rec = new CheckcodeRiskReq();
        rec.setMobile(baseRequest.getMobile());
        rec.setIp(IPUtil.getIpAdrress(request));
        rec.setDeviceIp(deviceIp);
        rec.setDeviceId(baseRequest.getDevice_token());
        rec.setSourceType(baseRequest.getSourceType());
        try {
            threadPoolTaskExecutor.execute(() -> {
                logger.info("recordRcData-execute-{}", JSON.toJSONString(rec));
                PicaResponse resp = rcClient.logSend(rec);
                logger.info("recordRcData-resp-{}", JSON.toJSONString(resp));
            });
        } catch (Exception e) {
            logger.error("recordRcData invoke error", e.getMessage());
        }
    }

    /**
     * @Description 拼图验证
     * @Author Chongwen.jiang
     * @Date 2020/3/23 15:04
     * @ModifyDate 2020/3/23 15:04
     * @Params [capCode, xPos]
     * @Return void
     */
    @Override
    public int validateJigsawCode(HttpServletRequest request, BaseRequest paramReq) {
        if(!SourceTypeEnum.H5.getCode().equals(paramReq.getSourceType())) {
            logger.error("validateJigsawCode-h5 sourceType error");
            throw new PicaWarnException(PicaResultCode.PARAM_IS_INVALID.code(), PicaResultCode.PARAM_IS_INVALID.message());
        }
        // 1、调用风控服务，是否需要触发拼图验证
         /*int needCheck = this.rcValidate(request, rcReq, "{}", paramReq.getSourceType());
        if (needCheck <= 0) {
            //  不需要拼图验证，直接去发短信
            return needCheck;
        }*/
        BaseRequest rcReq = new BaseRequest();
        rcReq.setMobile(paramReq.getMobile());
        String ip = IPUtil.getIpAdrress(request);
        String rcRecordedKey = Constants.RC_RECORDED_DATA_KEY.replace("{mobile}", paramReq.getMobile())
                .replace("{ip}", ip);
        logger.info("validateJigsawCode-rcRecordedKey-ip:{}, mobile：{}, key:{}", ip, paramReq.getMobile(), rcRecordedKey);
        int needCheck = 1;
        Long num = cacheClient.decr(rcRecordedKey);
        if (num != null && num.intValue() == 99) {
            //  不需要拼图验证，直接去发短信, 不用记录风控数据(在调用获取拼图码时已经校验过，记录了数据)
            return -1;
        }
        String capCode = paramReq.getCapCode();
        int xPos = paramReq.getxPos();
        if (StringUtils.isEmpty(capCode) || xPos <= 0) {
            String key = Constants.JIGSAW_CODE_KEY.replace("{mobile}", paramReq.getMobile());
            Long count = cacheClient.decr(key);
            if (count != null && count.intValue() == 99) {
                //  a、不需要拼图验证，可直接去发送短信
                //  b、本应需要拼图验证，但是在调用获取拼图数据接口时报错了，也直接去发送短信
                return needCheck;
            }
        }

        //  2、调用拼图验证接口
        PicaResponse resp = null;
        try {
            SliderImageRequest req = new SliderImageRequest();
            req.setCapcode(capCode);
            req.setXpos(xPos);
            resp = sliderImageService.checkcapcode(req);
            logger.info("validateJigsawCode-resp:{}", JSON.toJSONString(resp));
        } catch (Exception e) {
            logger.error("validateJigsawCode-sliderImageService.checkcapcode invoke exception", e.getMessage());
        }

        if (Objects.nonNull(resp)) {
            if (!PicaResultCode.SUCCESS.code().equals(resp.getCode())) {
                logger.info("validateJigsawCode-code exception");
            }
            JSONObject respData = JSON.parseObject(JSON.toJSONString(resp.getData()), JSONObject.class);
            if (Objects.nonNull(respData)) {
                Object codeObj = respData.get("code");
                if (Objects.nonNull(codeObj)) {
                    String code = String.valueOf(codeObj);
                    if (StringUtil.equals(code, "2") ||
                            StringUtil.equals(code, "3")) {
                        // 3超期 2验证失败 1成功
                        throw new PicaWarnException(AccountExceptionEnum.PICA_CAP_CODE_ERROR.getCode(),
                                AccountExceptionEnum.PICA_CAP_CODE_ERROR.getMessage());
                    }
                }
            }
        }

        return needCheck;
    }

    /**
     * @Description 获取拼图码校验
     * @Author Chongwen.jiang
     * @Date 2020/3/23 15:29
     * @ModifyDate 2020/3/23 15:29
     * @Params [mobile]
     * @Return com.pica.cloud.account.account.server.model.JigsawCodeModel
     */
    @Override
    public JigsawCodeModel getJigsawCode(HttpServletRequest request, Integer sourceType, String mobile) {
        if(!SourceTypeEnum.H5.getCode().equals(sourceType)) {
            logger.error("getJigsawCode-h5 sourceType error");
            throw new PicaWarnException(PicaResultCode.PARAM_IS_INVALID.code(), PicaResultCode.PARAM_IS_INVALID.message());
        }
        AccountUtils.checkMobilePhone(mobile);
        // 调用风控服务，是否需要触发拼图验证
        BaseRequest req = new BaseRequest();
        req.setMobile(mobile);
        int needCheck = this.rcValidate(request, req, "{}", sourceType);

        //  调用proof服务获取拼图数据
        JigsawCodeModel data = new JigsawCodeModel();
        PicaResponse picResp = null;
        String key = Constants.JIGSAW_CODE_KEY.replace("{mobile}", mobile);
        if (needCheck > 0) {
            try {
                picResp = sliderImageService.getPic();
                logger.info("getJigsawCode-getPic-code:{}", picResp.getCode());
            } catch (Exception e) {
                logger.error("getJigsawCode-sliderImageService.getPic invoke exception", e.getMessage());
                //  记录该手机号调用接口获取拼图时接口报错
                cacheClient.set(key, 100, Constants.JIGSAW_CODE_EXPIRE_SECONDS);
            }
            if (Objects.nonNull(picResp)) {
                if (!PicaResultCode.SUCCESS.code().equals(picResp.getCode())) {
                    logger.info("getJigsawCode-getPic-code exception");
                    cacheClient.set(key, 100, Constants.JIGSAW_CODE_EXPIRE_SECONDS);
                }
                JSONObject respData = JSON.parseObject(JSON.toJSONString(picResp.getData()), JSONObject.class);
                if (Objects.nonNull(respData)) {
                    logger.info("getJigsawCode-capcode:{}, yHeight:{}",
                            respData.getString("capcode"),
                            respData.getString("yHeight"));
                    data.setCapCode(respData.getString("capcode"));
                    data.setBackImage(respData.getString("backImage"));
                    data.setSlidingImage(respData.getString("slidingImage"));
                    data.setyHeight(respData.getInteger("yHeight"));
                }
            }
        } else {
            logger.info("getJigsawCode-don't need jigsaw code");
            cacheClient.set(key, 100, Constants.JIGSAW_CODE_EXPIRE_SECONDS);
            if (needCheck == -1) {
                //  不需要拼图验证, 风控允许直接发送验证码记录缓存
                String ip = IPUtil.getIpAdrress(request);
                String key2 = Constants.RC_RECORDED_DATA_KEY.replace("{mobile}", mobile)
                        .replace("{ip}", ip);
                logger.info("getJigsawCode-rcRecordedKey- rc recorded data-ip：{}，mobile：{}, rcRecordedKey:{}", ip, mobile, key2);
                cacheClient.set(key2, 100, Constants.JIGSAW_CODE_EXPIRE_SECONDS);
            }
        }

        return data;
    }


}
