package com.pica.cloud.account.account.server.service.impl;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.auth0.jwk.Jwk;
import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.entity.*;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.enums.SourceTypeEnum;
import com.pica.cloud.account.account.server.log.AccountLogEntityUtils;
import com.pica.cloud.account.account.server.log.AccountLogUtils;
import com.pica.cloud.account.account.server.mapper.*;
import com.pica.cloud.account.account.server.model.OneClickProcessor;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.req.OneClickLoginReq;
import com.pica.cloud.account.account.server.service.DoctorService;
import com.pica.cloud.account.account.server.service.LoginService;
import com.pica.cloud.account.account.server.service.RegisterService;
import com.pica.cloud.account.account.server.util.*;
import com.pica.cloud.foundation.completeness.client.utils.IntactUtils;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.entity.PicaWarnException;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.ValidateUtils;
import com.pica.cloud.patient.smartcontract.common.utils.HttpClientCloudUtils;
import io.jsonwebtoken.*;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.security.PublicKey;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

@Service
public class LoginServiceImpl implements LoginService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;

    @Autowired
    private AccountUnionMapper accountUnionMapper;

    @Autowired
    private AccountWeChatInfoMapper accountWeChatInfoMapper;

    @Autowired
    private AccountPatientInfoMapper accountPatientInfoMapper;

    @Autowired
    private AccountLogUtils picaLogUtils;

    @Autowired
    private TokenUtils tokenUtils;

    @Autowired
    private RegisterService registerService;

    @Autowired
    private DoctorMapper doctorInfoMapper;

    @Autowired
    private AccountUtils accountUtils;

    @Autowired
    private DoctorService doctorService;

    @Autowired
    private ICacheClient redisClient;

    @Autowired
    private AccountAppleInfoMapper accountAppleInfoMapper;

    @Autowired
    private OneClickProcessor oneClickProcessor;

    @Autowired
    private LogLoginAesMapper logLoginAesMapper;

    @Autowired
    private IntactUtil intactUtil;

    @Value("${doubleWritingMode}")
    private boolean doubleWritingMode;

    @Value("${weChatAppID}")
    private String appId;
    @Value("${weChatAppSecret}")
    private String appSecret;
    @Value("${weChatURL}")
    private String weChatURL;
    @Value("${weChatAppIDH5}")
    private String appIdH5;
    @Value("${weChatAppSecretH5}")
    private String appSecretH5;

    private static final String DEFAULT_DOCTOR_PICTURE_URL = "/File/doctor_default.png";
    public static Map<String, String> PIC_TYPE_MAP = new HashMap();

    static {
        PIC_TYPE_MAP.put("jpg", "FFD8FF");
        PIC_TYPE_MAP.put("jpeg", "FFD8FF");
        PIC_TYPE_MAP.put("png", "89504E47");
        PIC_TYPE_MAP.put("gif", "47494638");
        PIC_TYPE_MAP.put("bmp", "424D");
        PIC_TYPE_MAP.put("png", "89504E470D0a1a0a0000");
        PIC_TYPE_MAP.put("bmp", "424d228c010000000000");
        PIC_TYPE_MAP.put("bmp", "424d8240090000000000");
        PIC_TYPE_MAP.put("bmp", "424d8e1b030000000000");
    }

    @Override
    public LoginResult login(BaseRequest request) {
        //接入新旭事务一致性
        String batchNo = IntactUtils.getUUID();
        intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_1,"request:"+ JSON.toJSONString(request));
        String mobile = request.getMobile();
        Integer sourceType = request.getSourceType();
        String encrypt = AESUtil.encryptV0(mobile);
        AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(encrypt);
        logger.info("bizType:{}, account not null：{}", request.getBizType(), accountInfoEntity != null);
        if (accountInfoEntity != null) {
            String oldPwd = accountInfoEntity.getPassword();
            String password = request.getPassword();
            if (null != request.getBizType() &&
                    request.getBizType().equals(1)) {
                // 新版-未设置密码
                if (StringUtils.isEmpty(oldPwd)) {
                    if (sourceType != null && SourceTypeEnum.H5.getCode().equals(sourceType)) {
                        intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该手机号尚未设置密码，设置密码请前往云鹊医APP，或使用其他方式登录。");
                        throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_NOT_SETED_PASSWORD_H5.getCode(),
                                AccountExceptionEnum.PICA_MOBILE_NOT_SETED_PASSWORD_H5.getMessage());
                    } else {
                        intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该手机号{mobile}尚未设置密码，请先设置密码。");

                        throw new PicaException(AccountExceptionEnum.PICA_PASSWORD_RULE_ERROR.getCode(),
                                AccountExceptionEnum.PICA_PASSWORD_RULE_ERROR.getMessage()
                                        .replace("{mobile}", mobile));
                    }
                } else {
                    //  判断账号是否已锁
                    logger.info("new login failure：" + mobile);
                    String lockKey = Constants.ACCOUNT_LOCK_KEY.replace("{mobile}", mobile);
                    if (redisClient.exists(lockKey)) {
                        if (sourceType != null && SourceTypeEnum.H5.getCode().equals(sourceType)) {
                            intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该账号密码错误次数已达上限请24小时后再试，或请使用其他登录方式");

                            throw new PicaException(AccountExceptionEnum.PICA_PWD_MISMATCH_5_H5.getCode(),
                                    AccountExceptionEnum.PICA_PWD_MISMATCH_5_H5.getMessage());
                        } else {
                            intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该账号{mobile}的密码错误次数已达上限请24小时后再试，或请使用其他登录方式或找回密码");

                            throw new PicaException(AccountExceptionEnum.PICA_PWD_MISMATCH_5.getCode(),
                                    AccountExceptionEnum.PICA_PWD_MISMATCH_5.getMessage().replace("{mobile}", mobile));
                        }
                    }

                    if (password.equals(oldPwd)) {
                        intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"接下来调用pwdLoginCorrect");

                        return pwdLoginCorrect(request, mobile, encrypt, accountInfoEntity);
                    } else {
                        String errorKey = Constants.PWD_ERROR_NUM_KEY.replace("{mobile}", mobile);
                        if (redisClient.exists(errorKey)) {
                            int errorCount = Integer.parseInt(redisClient.get(errorKey));
                            errorCount = errorCount + 1;
                            redisClient.set(errorKey, errorCount, Constants.PWD_ERROR_NUM_SECONDS);
                            if (errorCount <= 4) {
                                intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"密码错误,请重试");

                                throw new PicaException(AccountExceptionEnum.PICA_PWD_MISMATCH_4.getCode(),
                                        AccountExceptionEnum.PICA_PWD_MISMATCH_4.getMessage());
                            } else {
                                //  设置账号锁定24h
                                redisClient.set(lockKey, mobile, Constants.PWD_ERROR_NUM_SECONDS);
                                if (sourceType != null && SourceTypeEnum.H5.getCode().equals(sourceType)) {
                                    intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该账号密码错误次数已达上限请24小时后再试，或请使用其他登录方式");

                                    throw new PicaException(AccountExceptionEnum.PICA_PWD_MISMATCH_5_H5.getCode(),
                                            AccountExceptionEnum.PICA_PWD_MISMATCH_5_H5.getMessage());
                                } else {
                                    intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该账号{mobile}的密码错误次数已达上限请24小时后再试，或请使用其他登录方式或找回密码");

                                    throw new PicaException(AccountExceptionEnum.PICA_PWD_MISMATCH_5.getCode(),
                                            AccountExceptionEnum.PICA_PWD_MISMATCH_5.getMessage().replace("{mobile}", mobile));
                                }
                            }
                        } else {
                            redisClient.set(errorKey, 1, Constants.PWD_ERROR_NUM_SECONDS);
                            intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"密码错误,请重试");

                            throw new PicaException(AccountExceptionEnum.PICA_PWD_MISMATCH_4.getCode(),
                                    AccountExceptionEnum.PICA_PWD_MISMATCH_4.getMessage());
                        }
                    }
                }

            } else {
                //  旧版本
                if (password.equals(oldPwd)) {
                    intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"接下来调用pwdLoginCorrect");

                    return pwdLoginCorrect(request, mobile, encrypt, accountInfoEntity);
                } else {
                    logger.info("login failure：" + mobile);
                    intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"请输入正确的密码");

                    throw new PicaException(AccountExceptionEnum.PICA_PASSWORD_ERROR.getCode(), AccountExceptionEnum.PICA_PASSWORD_ERROR.getMessage());
                }
            }
        } else {
            if (null != request.getBizType() &&
                    request.getBizType().equals(1)) {
                if (sourceType != null && SourceTypeEnum.H5.getCode().equals(sourceType)) {
                    intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该手机号尚未设置密码，设置密码请前往云鹊医APP，或使用其他方式登录。");

                    throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_NOT_REGIST_H5.getCode(),
                            AccountExceptionEnum.PICA_MOBILE_NOT_REGIST_H5.getMessage());
                } else {
                    //  新版-未注册
                    intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该手机号{mobile}尚未设置密码，请先设置密码。");

                    throw new PicaException(AccountExceptionEnum.PICA_PASSWORD_RULE_ERROR.getCode(),
                            AccountExceptionEnum.PICA_PASSWORD_RULE_ERROR.getMessage()
                                    .replace("{mobile}", mobile));
                }
            } else {
                intactUtil.sendIntact(batchNo,"login",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"未注册,请先注册");

                throw new PicaException(AccountExceptionEnum.PICA_NOT_REGISTER.getCode(), AccountExceptionEnum.PICA_NOT_REGISTER.getMessage());
            }
        }
    }

    /**
     * @Description 密码登录-密码正确逻辑
     * @Author Chongwen.jiang
     * @Date 2020/2/21 19:13
     * @ModifyDate 2020/2/21 19:13
     * @Params [request, mobile, encrypt, accountInfoEntity]
     * @Return com.pica.cloud.account.account.server.entity.LoginResult
     */
    private LoginResult pwdLoginCorrect(BaseRequest request, String mobile, String encrypt, AccountInfoEntity accountInfoEntity) {
        //接入新旭事务一致性
        String batchNo = IntactUtils.getUUID();
        intactUtil.sendIntact(batchNo,"pwdLoginCorrect",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_1,"request:"+ JSON.toJSONString(request)+",+mobile:"+mobile+",encrypt:"+encrypt+",accountInfoEntity:"+JSON.toJSONString(accountInfoEntity));
        Date currentTime = new Date();
        Integer acctId = accountInfoEntity.getId();
        int productType = request.getProductType();
        int sourceType = request.getSourceType();
        Doctor doctorInfo = doctorInfoMapper.getDoctorInfoByMobile(encrypt);
        Integer userId = 0;
        if (productType == AccountTypeEnum.PRODUCT_TYPE_DOCTOR.getCode()) {
            userId = doctorInfo.getId();
        } else if (productType == AccountTypeEnum.PRODUCT_TYPE_HEALTH.getCode()) {
            AccountPatientInfoEntity accountPatientInfoEntity = accountPatientInfoMapper.selectByAcctId(acctId);
            userId = accountPatientInfoEntity.getId();
        }
        Account account = new Account();
        account.setId(userId.longValue());
        account.setAcctId(acctId);
        account.setCreatTime(currentTime);
        account.setMobilePhone(mobile);
        account.setRegisterSource(sourceType);
        String newToken = tokenUtils.generateToken(account);
        LoginResult result = new LoginResult();
        result.setToken(newToken);
        result.setUserId(userId.longValue());
        result.setMobile(mobile);
        result.setDoctorId(EncryptUtils.encryptContent(userId + "", EncryptConstants.ENCRYPT_TYPE_ID));
        if (productType == AccountTypeEnum.PRODUCT_TYPE_DOCTOR.getCode()) {
            result.setEntireFlag(doctorInfo.getEntireFlag());

            LogLoginEntity entity = AccountLogEntityUtils.getLogLoginEntity(acctId, productType, sourceType,
                    AccountTypeEnum.LOGIN_PWD.getCode(), request.getLoginIp(), AccountTypeEnum.LOGIN_STATUS_SUCCESS.getCode(),
                    AccountTypeEnum.LOG_TYPE_LOGIN.getCode(), newToken, 1, request.getUserTokenTourist());
            picaLogUtils.info(entity);

            //  密码登录成功以后，清除错误次数记录
            String pwdErrorNum = Constants.PWD_ERROR_NUM_KEY.replace("{mobile}", mobile);
            if (redisClient.exists(pwdErrorNum)) {
                redisClient.del(pwdErrorNum);
            }
        }
        intactUtil.sendIntact(batchNo,"pwdLoginCorrect",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"request:"+ JSON.toJSONString(request)+",+mobile:"+mobile+",encrypt:"+",accountInfoEntity:"+JSON.toJSONString(accountInfoEntity));

        return result;
    }


        @Override
        public LoginResult loginAndRegister(BaseRequest baseRequest) {
            String mobile = baseRequest.getMobile();
            AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(AESUtil.encryptV0(mobile));
            logger.info("loginAndRegister-accountInfoEntity is null {}, clientIp:{}", (accountInfoEntity == null), baseRequest.getLoginIp());
            if (accountInfoEntity == null) {
                //说明是注册功能
                accountUtils.checkRegisterMobilePhoneAndAuthCode(baseRequest.getMobile(), baseRequest.getFlag() + "", baseRequest.getAuthCode());
                return registerService.register(baseRequest);
            } else {
                //登录功能
                accountUtils.checkMobilePhoneAndAuthCode(baseRequest.getMobile(), AccountTypeEnum.SYSCODE_TYPE_LOGIN.getCode() + "", baseRequest.getAuthCode());
                return processLogin(baseRequest, accountInfoEntity.getId(), AccountTypeEnum.LOGIN_CODE.getCode());
            }
        }

        /**
         * 登录逻辑处理
         * 登录逻辑处理
         *
         * @param baseRequest
         */
        private LoginResult processLogin(BaseRequest baseRequest, Integer acctId, Integer loginType) {
            return processLogin(baseRequest, acctId, loginType, null);
        }

        private LoginResult processLogin(BaseRequest baseRequest, Integer acctId, Integer loginType, QueryMobileEntity queryMobileEntity) {
            //接入新旭事务一致性
            String batchNo = IntactUtils.getUUID();
            intactUtil.sendIntact(batchNo,"processLogin",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_1,"baseRequest:"+ JSON.toJSONString(baseRequest)+",+acctId:"+acctId+",loginType:"+loginType+",queryMobileEntity:"+JSON.toJSONString(queryMobileEntity));
            Date currentTime = new Date();
            Long userId = accountUtils.getUserIdByAcctId(baseRequest.getProductType(), acctId);
            Account account = new Account();
            account.setId(userId);
            account.setAcctId(acctId);
            account.setCreatTime(currentTime);
            account.setMobilePhone(baseRequest.getMobile());
            account.setRegisterSource(baseRequest.getSourceType());
            String newToken = tokenUtils.generateToken(account);
            LoginResult result = new LoginResult();
            result.setToken(newToken);
            result.setUserId(userId);
            result.setMobile(baseRequest.getMobile());
            result.setDoctorId(EncryptUtils.encryptContent(userId + "", EncryptConstants.ENCRYPT_TYPE_ID));
            //是否完善过个人信息（云鹊医app才需要）
            if (baseRequest.getProductType() == AccountTypeEnum.PRODUCT_TYPE_DOCTOR.getCode()) {
                Doctor doctorEntity = doctorInfoMapper.selectByPrimaryKey(userId.intValue());
                result.setEntireFlag(doctorEntity.getEntireFlag());
            }
            //记录登录日志
            LogLoginEntity entity = AccountLogEntityUtils.getLogLoginEntity(acctId, baseRequest.getProductType(), baseRequest.getSourceType(),
                    loginType, baseRequest.getLoginIp(), AccountTypeEnum.LOGIN_STATUS_SUCCESS.getCode(), AccountTypeEnum.LOG_TYPE_LOGIN.getCode(),newToken,1,baseRequest.getUserTokenTourist());
            if (queryMobileEntity != null) {
                entity.setQueryMobileEntity(queryMobileEntity);
            }
            picaLogUtils.info(entity);
            intactUtil.sendIntact(batchNo,"processLogin",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"baseRequest:"+ JSON.toJSONString(baseRequest)+",+acctId:"+acctId+",loginType:"+loginType+",queryMobileEntity:"+JSON.toJSONString(queryMobileEntity));

            return result;
        }

        @Override
        @Transactional
        public LoginResult loginByWeChat(BaseRequest request) {
            //接入新旭事务一致性
            String batchNo = IntactUtils.getUUID();
            intactUtil.sendIntact(batchNo,"loginByWeChat",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_1,"request:"+ JSON.toJSONString(request));
            WeChatEntity weChatEntity = null;
            if (null != request.getBizType() &&
                    request.getBizType().equals(1)) {
                //  h5微信登录 TODO 下次单独拆分出去一个新接口
                weChatEntity = WeChatUtils.getAuthorizationInfo(appIdH5, appSecretH5, request.getWeChatCode());
            } else {
                //  原生微信登录
                weChatEntity = WeChatUtils.getAuthorizationInfo(appId, appSecret, request.getWeChatCode());
            }
            if (weChatEntity == null || StringUtils.isEmpty(weChatEntity.getOpenid()) || StringUtils.isEmpty(weChatEntity.getAccess_token())) {
                intactUtil.sendIntact(batchNo,"loginByWeChat",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"微信登录授权code不正确:");

                throw new PicaException(AccountExceptionEnum.PICA_WECHAT_CODE_ERROR.getCode(), AccountExceptionEnum.PICA_WECHAT_CODE_ERROR.getMessage());
            }
            //  微信登录获取个人信息
            Map map = new HashMap();
            map.put("access_token", weChatEntity.getAccess_token());
            map.put("openid", weChatEntity.getOpenid());
            Map weChatUserInfo = WeChatUtils.getWeChatUserInfo(map, weChatURL);
            WeChatUserInfoEntity weChatUserInfoEntity = WeChatUtils.mergeWechatUserInfo(weChatUserInfo, weChatEntity.getOpenid());
            String unionId = weChatUserInfoEntity.getUnionid();
            AccountUnionEntity accountUnionEntity = accountUnionMapper.selectByUnionId(unionId);
            //是否绑定逻辑的判断
            if (accountUnionEntity != null) {
                Long acctId = accountUnionEntity.getAcctId();
                Long userId = accountUtils.getUserIdByAcctId(request.getProductType(), acctId.intValue());
                Account account = new Account();
                account.setId(userId);
                account.setAcctId(acctId.intValue());
                account.setCreatTime(new Date());
                //account.setMobilePhone(request.getMobile());
                account.setMobilePhone("");
                account.setRegisterSource(request.getSourceType());
                String newToken = tokenUtils.generateToken(account);
                LoginResult result = new LoginResult();
                result.setToken(newToken);
                result.setUserId(userId);
                result.setBindFlag(AccountTypeEnum.BIND_STATUS_SUCCESS.getCode() + "");
                result.setDoctorId(EncryptUtils.encryptContent(userId + "", EncryptConstants.ENCRYPT_TYPE_ID));
                if (request.getProductType() == AccountTypeEnum.PRODUCT_TYPE_DOCTOR.getCode()) {
                    PICAPDoctor doctor = doctorInfoMapper.queryDoctor(userId);
                    result.setEntireFlag(doctor.getEntire_flag());
                    result.setMobile(AESUtil.decryptV0(doctor.getMobile_phone()));
                }

                //记录登录日志
                LogLoginEntity entity = AccountLogEntityUtils.getLogLoginEntity(
                        Integer.valueOf(String.valueOf(acctId)),
                        request.getProductType(), request.getSourceType(),
                        AccountTypeEnum.LOGIN_WE_CHAT.getCode(),
                        request.getLoginIp(),
                        AccountTypeEnum.LOGIN_STATUS_SUCCESS.getCode(),
                        AccountTypeEnum.LOG_TYPE_LOGIN.getCode(),newToken,1,request.getUserTokenTourist());
                picaLogUtils.info(entity);
                intactUtil.sendIntact(batchNo,"loginByWeChat",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"request:"+ JSON.toJSONString(request));


                return result;
            } else {
                AccountWeChatInfoEntity entity = accountWeChatInfoMapper.selectByUnionId(unionId);
                //如果微信信息表数据不存在，就把用户信息存储到微信信息表中。
                if (entity == null) {
                    processWeChatInfoUser(weChatUserInfoEntity, request.getWeChatLoginType());
                }
                LoginResult result = new LoginResult();
                result.setUnionId(unionId);
                result.setBindFlag(AccountTypeEnum.BIND_STATUS_FAILURE.getCode() + "");
                intactUtil.sendIntact(batchNo,"loginByWeChat",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"request:"+ JSON.toJSONString(request));

                return result;
            }
        }

        @Override
        public LoginResult loginByWeChatStep(BaseRequest request) {
            //判断当前手机号是否注册过,注册过，直接登录；没有注册过，进行注册操操作
            AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(AESUtil.encryptV0(request.getMobile()));
            logger.info("loginByWeChatStep-accountInfoEntity is null {}, clientIp:{}", (accountInfoEntity == null), request.getLoginIp());
            LoginResult result;
            if (accountInfoEntity == null) {
                accountUtils.checkRegisterMobilePhoneAndAuthCode(
                        request.getMobile(),
                        AccountTypeEnum.SYSCODE_TYPE_WE_CHAT.getCode() + "",
                        request.getAuthCode());
                result = registerService.register(request);
                if (doubleWritingMode) {
                    //双写模式下,要在doctor表存储unionId
                    if (result.getUserId() != null) {
                        Doctor doctor = new Doctor();
                        doctor.setId(result.getUserId().intValue());
                        doctor.setUnionid(request.getUnionId());
                        doctorInfoMapper.updateByPrimaryKeySelective(doctor);
                    }
                }
            } else {
                accountUtils.checkMobilePhoneAndAuthCode(
                        request.getMobile(),
                        AccountTypeEnum.SYSCODE_TYPE_WE_CHAT.getCode() + "",
                        request.getAuthCode());
                result = processLogin(request, accountInfoEntity.getId(), AccountTypeEnum.LOGIN_WE_CHAT.getCode());
            }
            result.setMobile(request.getMobile());
            AccountInfoEntity accountInfo = accountInfoDetailMapper.selectByMobile(AESUtil.encryptV0(request.getMobile()));
            Integer acctId = accountInfo.getId();
            processAccountUnion(acctId, request.getUnionId(), request.getProductType());
            return result;
        }

        @Override
        @Transactional
        public void unbindWeChat(long doctorId) {
            Integer acctId = doctorInfoMapper.getAcctIdByDoctorId(doctorId);
            Map<String, Object> map = new HashedMap(2);
            map.put("acctId", acctId);
            map.put("unionType", AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());
            accountUnionMapper.updateUnbindByAcctId(map);
            if (doubleWritingMode) {
                doctorService.unbindWeChat(acctId);
            }
        }

        @Override
        @Transactional
        public String bindWeChat(long doctorId, BaseRequest request) {
            WeChatEntity weChatEntity = WeChatUtils.getAuthorizationInfo(appId, appSecret, request.getWeChatCode());
            Map map = new HashMap();
            map.put("access_token", weChatEntity.getAccess_token());
            map.put("openid", weChatEntity.getOpenid());
            Map weChatUserInfo = WeChatUtils.getWeChatUserInfo(map, weChatURL);
            WeChatUserInfoEntity weChatUserInfoEntity = WeChatUtils.mergeWechatUserInfo(weChatUserInfo, weChatEntity.getOpenid());
            String unionId = weChatUserInfoEntity.getUnionid();
            AccountWeChatInfoEntity entity = accountWeChatInfoMapper.selectByUnionId(unionId);
            if (entity == null) {
                processWeChatInfoUser(weChatUserInfoEntity, request.getWeChatLoginType());
            } else {
                updateWechatInfoUser(entity, weChatUserInfoEntity);
            }
            Integer acctId = doctorInfoMapper.getAcctIdByDoctorId(doctorId);
            processAccountUnion(acctId, unionId, request.getProductType());
            return weChatUserInfoEntity.getNickname();
        }

        /**
         * 把unionId存储到联合登录表中
         *
         * @param acctId
         * @param unionId
         */
        private void processAccountUnion(Integer acctId, String unionId, Integer productType) {
            //接入新旭事务一致性
            String batchNo = IntactUtils.getUUID();
            intactUtil.sendIntact(batchNo,"processAccountUnion",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_1,"acctId:"+acctId+",unionId:"+unionId+",productType:"+productType);
            //先查询当前产品线是否有记录？如果有就更新成delete_flag=2; 然后在插入新的绑定记录
            AccountUnionEntity accountUnionResult = accountUnionMapper.selectByUnionId(unionId);
            if (accountUnionResult != null) {
                intactUtil.sendIntact(batchNo,"processAccountUnion",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该微信号已绑定其他云鹊医账户，你可以使用微信登录云鹊医，在「设置」页解除绑定");

                throw new PicaException(AccountExceptionEnum.PICA_WECHAT_UNBIND.getCode(), AccountExceptionEnum.PICA_WECHAT_UNBIND.getMessage());
            }
            Map<String, Object> map = new HashedMap(2);
            map.put("acctId", acctId);
            map.put("unionType", AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());
            AccountUnionEntity accountUnionEntityAccount = accountUnionMapper.selectByAcctId(map);
            if (accountUnionEntityAccount != null) {
                intactUtil.sendIntact(batchNo,"processAccountUnion",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"该手机号已绑定其他微信号，你可以在「设置」页解除绑定");

                throw new PicaException(AccountExceptionEnum.PICA_WECHAT_BIND_OTHER.getCode(), AccountExceptionEnum.PICA_WECHAT_BIND_OTHER.getMessage());
            }
            AccountUnionEntity accountUnionEntity = new AccountUnionEntity();
            accountUnionEntity.setAcctId(acctId.longValue());
            accountUnionEntity.setDeleteFlag(1);
            accountUnionEntity.setUnionId(unionId);
            accountUnionEntity.setCreatedTime(new Date());
            accountUnionEntity.setModifiedTime(new Date());
            accountUnionEntity.setCreatedId(acctId);
            accountUnionEntity.setModifiedId(acctId);
            accountUnionEntity.setUnionType(AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());
            accountUnionMapper.insertSelective(accountUnionEntity);
            if (doubleWritingMode) {  //双写模式
                doctorService.bindWeChat(acctId, unionId);
            }
            intactUtil.sendIntact(batchNo,"processAccountUnion",com.pica.cloud.foundation.completeness.contract.constants.CommonConstants.INTACT_CONTENT_LOG_STATUS_3,"acctId:"+acctId+",unionId:"+unionId+",productType:"+productType);

        }

        private void processWeChatInfoUser(WeChatUserInfoEntity weChatUserInfoEntity, int type) {
            AccountWeChatInfoEntity accountWeChatInfoEntity = new AccountWeChatInfoEntity();
            Date currentTime = new Date();
            accountWeChatInfoEntity.setCreatedId(0);
            accountWeChatInfoEntity.setCreatedTime(currentTime);
            accountWeChatInfoEntity.setDeleteFlag(1);
            accountWeChatInfoEntity.setGroupid(weChatUserInfoEntity.getGroupid() + "");
            accountWeChatInfoEntity.setType(type);
            accountWeChatInfoEntity.setModifiedId(0);
            accountWeChatInfoEntity.setModifiedTime(currentTime);
            accountWeChatInfoEntity.setOpenid(weChatUserInfoEntity.getOpenid());
            accountWeChatInfoEntity.setUnionid(weChatUserInfoEntity.getUnionid());
            accountWeChatInfoEntity.setPrivilege(weChatUserInfoEntity.getPrivilege());
            accountWeChatInfoEntity.setRemark(weChatUserInfoEntity.getRemark());
            accountWeChatInfoEntity.setSubscribe(weChatUserInfoEntity.getSubscribe());
            accountWeChatInfoEntity.setSubscribeTime(weChatUserInfoEntity.getSubscribe_time());
            accountWeChatInfoEntity.setTagidList(weChatUserInfoEntity.getTagid_list());
            accountWeChatInfoEntity.setCity(weChatUserInfoEntity.getCity());
            accountWeChatInfoEntity.setNickname(weChatUserInfoEntity.getNickname());
            accountWeChatInfoEntity.setHeadImgUrl(weChatUserInfoEntity.getHeadimgurl());
            accountWeChatInfoEntity.setCountry(weChatUserInfoEntity.getCountry());
            accountWeChatInfoEntity.setSex(weChatUserInfoEntity.getSex());
            accountWeChatInfoEntity.setProvince(weChatUserInfoEntity.getProvince());
            accountWeChatInfoEntity.setLanguage(weChatUserInfoEntity.getLanguage());
            accountWeChatInfoMapper.insertSelective(accountWeChatInfoEntity);
        }

        private void updateWechatInfoUser(AccountWeChatInfoEntity entity, WeChatUserInfoEntity weChatUserInfoEntity) {
            String nickname = weChatUserInfoEntity.getNickname();
            if (StringUtils.isNotEmpty(nickname) && !nickname.equals(entity.getNickname())) {
                AccountWeChatInfoEntity info = new AccountWeChatInfoEntity();
                info.setId(entity.getId());
                info.setNickname(nickname);
                accountWeChatInfoMapper.updateByPrimaryKeySelective(info);
            }
        }

        @Override
        public PICAPDoctor queryDoctor(long doctorId) {
            PICAPDoctor doctor = doctorInfoMapper.queryDoctor(doctorId);
            if (doctor == null) {
                doctor = new PICAPDoctor();
            }
            doctor.setAvatar_image_url(this.processDoctorAvatar(doctor.getAvatar_image_url()));
            return doctor;
        }

        @Override
        public LoginResult oneClickLogin(OneClickLoginReq req) {
            QueryMobileEntity queryMobileEntity = oneClickProcessor.tokenExchangeMobile(req.getToken(), req.getSourceType());
            if (queryMobileEntity == null || StringUtils.isBlank(queryMobileEntity.getMobile()) || !ValidateUtils.isMobile(queryMobileEntity.getMobile())) {
                logger.error("oneClickLogin 闪验获取出错-queryMobileEntity:{}",JSONObject.toJSONString(queryMobileEntity));
                throw new PicaException(PicaResultCode.INTERFACE_INVOKE_EXCEPTION.code(), "获取手机号失败！");
            }
            queryMobileEntity.setSourceType(req.getSourceType());
            AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(AESUtil.encryptV0(queryMobileEntity.getMobile()));
            BaseRequest baseRequest = new BaseRequest();
            baseRequest.setMobile(queryMobileEntity.getMobile());
            baseRequest.setSourceType(req.getSourceType());
            baseRequest.setProductType(req.getProductType());
            baseRequest.setLoginIp(req.getLoginIp());
            baseRequest.setUserTokenTourist(req.getUserTokenTourist());
            LoginResult result;
            if (accountInfoEntity == null) {
                //说明是注册功能
                result = registerService.register(baseRequest, queryMobileEntity);
            } else {
                //登录功能
                result = processLogin(baseRequest, accountInfoEntity.getId(), AccountTypeEnum.LOGIN_ONE_CLICK.getCode(), queryMobileEntity);
            }
            return result;
        }


        private String processDoctorAvatar(String avatar_image_url) {
            if (StringUtils.isEmpty(avatar_image_url)) {
                return DEFAULT_DOCTOR_PICTURE_URL;
            } else {
                int pos = avatar_image_url.lastIndexOf(".");
                if (pos < 0) {
                    return DEFAULT_DOCTOR_PICTURE_URL;
                } else {
                    String ext = avatar_image_url.substring(pos + 1, avatar_image_url.length()).toLowerCase();
                    return !PIC_TYPE_MAP.containsKey(ext) ? DEFAULT_DOCTOR_PICTURE_URL : avatar_image_url;
                }
            }
        }

        /**
         * @Description 统一校验(传空则不会校验)
         * @Author Chongwen.jiang
         * @Date 2020/2/20 16:55
         * @ModifyDate 2020/2/20 16:55
         * @Params [request]
         * @Return com.pica.cloud.foundation.entity.PicaResponse
         */
        @Override
        public void preLoginValidate(BaseRequest request) {
            Integer bizType = request.getBizType();
            String mobile = request.getMobile();
            Integer sourceType = request.getSourceType();

            if (null == bizType) {
                logger.info("bizType is null");
                throw new PicaWarnException(PicaResultCode.PARAM_IS_BLANK.code(),
                        PicaResultCode.PARAM_IS_BLANK.message());
            }

            if (bizType.equals(2)) {
                //  手机号规则校验
                if (StringUtils.isNotEmpty(mobile) &&
                        !ValidateUtils.isMobile(mobile)) {
                    throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_REG_FALSE.getCode(),
                            AccountExceptionEnum.PICA_MOBILE_REG_FALSE.getMessage());
                }
            } else if (bizType.equals(3)) {
                //  手机号规则校验
                if (StringUtils.isNotEmpty(mobile) &&
                        !ValidateUtils.isMobile(mobile)) {
                    throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_REG_FALSE.getCode(),
                            AccountExceptionEnum.PICA_MOBILE_REG_FALSE.getMessage());
                }
                //  手机号是否注册
                AccountInfoEntity accountInfo = accountInfoDetailMapper.selectByMobile(AESUtil.encryptV0(mobile));
                if (accountInfo == null) {
                    if (sourceType != null && SourceTypeEnum.H5.getCode().equals(sourceType)) {
                        throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_NOT_REGIST_H5.getCode(),
                                AccountExceptionEnum.PICA_MOBILE_NOT_REGIST_H5.getMessage());
                    } else {
                        throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_NOT_REGIST.getCode(),
                                AccountExceptionEnum.PICA_MOBILE_NOT_REGIST.getMessage().replace("{mobile}", mobile));
                    }
                }
                //  手机号是否设置了密码
                if (StringUtils.isEmpty(accountInfo.getPassword())) {
                    if (sourceType != null && SourceTypeEnum.H5.getCode().equals(sourceType)) {
                        throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_NOT_SETED_PASSWORD_H5.getCode(),
                                AccountExceptionEnum.PICA_MOBILE_NOT_SETED_PASSWORD_H5.getMessage());
                    } else {
                        throw new PicaWarnException(AccountExceptionEnum.PICA_MOBILE_NOT_SETED_PASSWORD.getCode(),
                                AccountExceptionEnum.PICA_MOBILE_NOT_SETED_PASSWORD.getMessage().replace("{mobile}", mobile));
                    }
                }
            }


        }

        /**
         * @Description 苹果登录授权
         * @Author Chongwen.jiang
         * @Date 2020/2/24 19:21
         * @ModifyDate 2020/2/24 19:21
         * @Params [request]
         * @Return com.pica.cloud.account.account.server.entity.LoginResult
         */
        @Transactional
        @Override
        public LoginResult loginByApple(BaseRequest request) {
            long start = System.currentTimeMillis();
            //  入参非空判断
            if (StringUtils.isEmpty(request.getIdentifyToken())) {
                throw new PicaException(AccountExceptionEnum.PICA_PARAMS_ERROR.getCode(),
                        AccountExceptionEnum.PICA_PARAMS_ERROR.getMessage());
            }
            //  identifyToken校验
            if(!checkIdentifyToken(request)){
                throw new PicaException(
                        AccountExceptionEnum.PICA_APPLE_TOKEN_ERROR.getCode(),
                        AccountExceptionEnum.PICA_APPLE_TOKEN_ERROR.getMessage());
            }
            //  apple授权登录用户信息入库
            AccountUnionEntity accountUnion = accountUnionMapper.selectByUnionId(request.getAppleUserId());
            //是否绑定逻辑的判断
            if (accountUnion != null) {
                Long acctId = accountUnion.getAcctId();
                Long userId = accountUtils.getUserIdByAcctId(null, acctId.intValue());
                Account account = new Account();
                account.setId(userId);
                account.setAcctId(acctId.intValue());
                account.setCreatTime(new Date());
                account.setMobilePhone("");
                account.setRegisterSource(request.getSourceType());
                String newToken = tokenUtils.generateToken(account);
                LoginResult result = new LoginResult();
                result.setToken(newToken);
                result.setUserId(userId);
                result.setBindFlag(AccountTypeEnum.BIND_STATUS_SUCCESS.getCode() + "");
                result.setDoctorId(EncryptUtils.encryptContent(userId + "", EncryptConstants.ENCRYPT_TYPE_ID));
                if (request.getProductType() == AccountTypeEnum.PRODUCT_TYPE_DOCTOR.getCode()) {
                    PICAPDoctor doctor = doctorInfoMapper.queryDoctor(userId);
                    result.setEntireFlag(doctor.getEntire_flag());
                    result.setMobile(AESUtil.decryptV0(doctor.getMobile_phone()));
                }
                //记录登录日志
                LogLoginEntity entity = AccountLogEntityUtils.getLogLoginEntity(
                        Integer.valueOf(String.valueOf(acctId)),
                        request.getProductType(), request.getSourceType(),
                        AccountTypeEnum.LOGIN_APPLE.getCode(),
                        request.getLoginIp(),
                        AccountTypeEnum.LOGIN_STATUS_SUCCESS.getCode(),
                        AccountTypeEnum.LOG_TYPE_LOGIN.getCode(),newToken,1,request.getUserTokenTourist());
                picaLogUtils.info(entity);

                long end1 = System.currentTimeMillis();
                logger.info("loginByApple1-duration {} millionSeconds", (end1-start));
                return result;
            } else {
                AccountAppleInfo entity = accountAppleInfoMapper.selectByUserId(request.getAppleUserId());
                //如果apple信息表数据不存在，就把用户信息存储到apple信息表中。
                if (entity == null) {
                    processAppleInfoUser(request);
                }
                LoginResult result = new LoginResult();
                result.setUnionId(request.getAppleUserId());
                result.setBindFlag(AccountTypeEnum.BIND_STATUS_FAILURE.getCode() + "");
                long end1 = System.currentTimeMillis();
                logger.info("loginByApple2-duration {} millionSeconds", (end1-start));
                return result;
            }
        }

        /**
         * @Description apple登录--identifyToken校验
         * @Author Chongwen.jiang
         * @Date 2020/2/24 19:28
         * @ModifyDate 2020/2/24 19:28
         * @Params [identifyToekn]
         * @Return boolean false:未通过token校验，true:通过校验
         */
        private boolean checkIdentifyToken(BaseRequest request) {
            String identifyToken = request.getIdentifyToken();
            logger.info("checkIdentifyToken-identifyToken:{}", identifyToken);
            //  向苹果后台获取公钥参数
            String appleResp = null;
            try {
                appleResp = HttpClientCloudUtils.getHttpExecute(Constants.APPLE_GET_PUBLIC_KEY_URL);
                logger.info("checkIdentifyToken-appleResp:{}", appleResp);
            } catch (Exception e) {
                logger.info("checkIdentifyToken-get apple public key fail " + e.getMessage());
                throw new PicaException("get apple public key fail Exception", "get apple public key fail");
            }
            JSONObject appleRespJson = JSONObject.parseObject(appleResp);
            String keys = appleRespJson.getString("keys");
            JSONArray keysArr = JSONObject.parseArray(keys);

            if (identifyToken.split("\\.").length < 2) {
                throw new PicaException("get identifyToken fail Exception", "get identifyToken format Exception");
            }
            JSONObject useAppleAuth = new JSONObject();
            String inAuth = new String(Base64.decodeBase64(identifyToken.split("\\.")[0]));
            String inKid = JSONObject.parseObject(inAuth).get("kid").toString();
            for(Object obj : keysArr){
                JSONObject appleAuth = JSONObject.parseObject(obj.toString());
                if(inKid.equals(appleAuth.getString("kid"))){
                    useAppleAuth = appleAuth;
                    logger.info("checkIdentifyToken-jsonObject1:{}", useAppleAuth);
                    break;
                }
            }

            //  通过jar生成publicKey
            PublicKey publicKey;
            try {
                Jwk jwa = Jwk.fromValues(useAppleAuth);
                publicKey = jwa.getPublicKey();
            } catch (Exception e) {
                logger.info("checkIdentifyToken-generate publicKey fail " + e.getMessage());
                throw new PicaException("checkIdentifyToken-generate publicKey fail", "generate publicKey fail");
            }

            //  分割前台传过来的identifyToken（jwt格式的token）用base64解码使用
            String aud;
            String sub;
            try {
                String claim = new String(Base64.decodeBase64(identifyToken.split("\\.")[1]));
                //logger.info("checkIdentifyToken-claim:{}", claim);
                aud = JSONObject.parseObject(claim).get("aud").toString();
                sub = JSONObject.parseObject(claim).get("sub").toString();
                //  appleUserId从token中解码取出后赋值
                request.setAppleUserId(sub);
            } catch (Exception e) {
                logger.info("checkIdentifyToken-token decode fail " + e.getMessage());
                throw new PicaException("checkIdentifyToken-token decode fail Exception", "token decode fail");
            }
            return this.verify(publicKey, identifyToken, aud, sub, request);
        }

        /**
         * @Description 验证苹果公钥
         * @Author Chongwen.jiang
         * @Date 2020/2/24 19:49
         * @ModifyDate 2020/2/24 19:49
         * @Params [key, jwt, audience, subject]
         * @Return boolean
         */
        private boolean verify(PublicKey key, String jwt, String audience, String subject, BaseRequest request) {
            JwtParser jwtParser = Jwts.parser().setSigningKey(key);
            jwtParser.requireIssuer(Constants.APPLE_ISSUE_URL);
            jwtParser.requireAudience(audience);
            jwtParser.requireSubject(subject);
            try {
                logger.info("checkIdentifyToken-apple-verify-starting");
                Jws<Claims> claim = jwtParser.parseClaimsJws(jwt);
                logger.info("acheckIdentifyToken-apple-verify-claim:{}", JSON.toJSONString(claim));
                //logger.info("apple-verify-claim.getBody:{}", JSON.toJSONString(claim.getBody()));
                if (claim != null && claim.getBody().containsKey("auth_time")) {
                    request.setInfo(JSON.toJSONString(claim.getBody()));
                    JSONObject claimBody = JSONObject.parseObject(JSON.toJSONString(claim.getBody()), JSONObject.class);
                    request.setAppleId(claimBody.getString("email"));
                    return true;
                }
                return false;
            } catch (ExpiredJwtException e) {
                logger.info("checkIdentifyToken-apple token expired " + e.getMessage());
                throw new PicaException("apple token expired Exception", "apple token expired");
            } catch (Exception e) {
                logger.info("checkIdentifyToken-apple token illegal " + e.getMessage());
                throw new PicaException("apple token illegal Exception", "apple token illegal");
            }
        }

        /**
         * @Description apple用户信息入表
         * @Author Chongwen.jiang
         * @Date 2020/2/24 11:00
         * @ModifyDate 2020/2/24 11:00
         * @Params [request]
         * @Return void
         */
        private void processAppleInfoUser(BaseRequest request) {
            AccountAppleInfo appleInfo = new AccountAppleInfo();
            appleInfo.setCreatedId(0);
            appleInfo.setModifiedId(0);
            appleInfo.setAppleUserId(request.getAppleUserId());
            appleInfo.setAppleId(request.getAppleId());
            appleInfo.setInfo(request.getInfo());
            accountAppleInfoMapper.insertSelective(appleInfo);
        }

        /**
         * @Description 苹果登录绑定手机号
         * @Author Chongwen.jiang
         * @Date 2020/2/24 11:40
         * @ModifyDate 2020/2/24 11:40
         * @Params [request]
         * @Return com.pica.cloud.account.account.server.entity.LoginResult
         */
        @Override
        public LoginResult loginByAppleStep(BaseRequest request) {
            //  判断当前手机号是否注册过: 没有注册过，进行注册操操作, 注册过，直接登录；
            AccountInfoEntity accountInfoDb = accountInfoDetailMapper.selectByMobile(AESUtil.encryptV0(request.getMobile()));
            logger.info("loginByAppleStep-account is null {}, request:{}", accountInfoDb == null, JSON.toJSONString(request));
            LoginResult result;
            if (accountInfoDb == null) {
                //  验证码校验
                accountUtils.checkRegisterMobilePhoneAndAuthCode(
                        request.getMobile(),
                        AccountTypeEnum.SYSCODE_TYPE_APPLE.getCode() + "",
                        request.getAuthCode());
                result = registerService.register(request);
                logger.info("loginByAppleStep-register");
            } else {
                //  验证码校验
                accountUtils.checkMobilePhoneAndAuthCode(request.getMobile(),
                        AccountTypeEnum.SYSCODE_TYPE_APPLE.getCode() + "",
                        request.getAuthCode());
                result = processLogin(request, accountInfoDb.getId(),
                        AccountTypeEnum.LOGIN_APPLE.getCode());
                logger.info("loginByAppleStep-processLogin");
            }
            result.setMobile(request.getMobile());
            AccountInfoEntity accountInfo = accountInfoDetailMapper.selectByMobile(AESUtil.encryptV0(request.getMobile()));
            Integer acctId = accountInfo.getId();
            //  insert account_apple_info表数据
            logger.info("loginByAppleStep-insert-account_apple_info-start");
            processAccountUnionApple(acctId, request.getAppleUserId());
            logger.info("loginByAppleStep-insert-account_apple_info-end");
            return result;
        }

        /**
         * @Description account_apple_info insert
         * @Author Chongwen.jiang
         * @Date 2020/2/24 11:34
         * @ModifyDate 2020/2/24 11:34
         * @Params [acctId, appleUserId]
         * @Return void
         */
        private void processAccountUnionApple(Integer acctId, String appleUserId) {
            AccountUnionEntity accountUnion = accountUnionMapper.selectByUnionId(appleUserId);
            if (accountUnion != null) {
                throw new PicaException(
                        AccountExceptionEnum.PICA_APPLE_BIND_OTHER.getCode(),
                        AccountExceptionEnum.PICA_APPLE_BIND_OTHER.getMessage());
            }
            Map<String, Object> map = new HashedMap(2);
            map.put("acctId", acctId);
            map.put("unionType", AccountTypeEnum.UNION_LOGIN_APPLE.getCode());
            AccountUnionEntity accountUnionEntityAccount = accountUnionMapper.selectByAcctId(map);
            if (accountUnionEntityAccount != null) {
                throw new PicaException(
                        AccountExceptionEnum.PICA_MOBILE_BIND_OTHER.getCode(),
                        AccountExceptionEnum.PICA_MOBILE_BIND_OTHER.getMessage());
            }
            AccountUnionEntity accountUnionEntity = new AccountUnionEntity();
            accountUnionEntity.setAcctId(acctId.longValue());
            accountUnionEntity.setUnionId(appleUserId);
            accountUnionEntity.setCreatedId(acctId);
            accountUnionEntity.setModifiedId(acctId);
            accountUnionEntity.setUnionType(AccountTypeEnum.UNION_LOGIN_APPLE.getCode());
            accountUnionMapper.insertSelective(accountUnionEntity);
        }


        /**
         * @Description TODO
         * @Author peijun.zhao
         * @Date 2020/4/22 17:14
         * @ModifyDate 2020/4/22 17:14
         * @Params [aes]
         * @Return int
         */
        @Override
        public int insertLoginAesLog(LogLoginAes aes){
            aes.setDeleteFlag(1);
            aes.setCreateId(999999);
            aes.setModifyId(999999);
            aes.setCreateTime(new Date());
            aes.setModifyTime(new Date());
            return logLoginAesMapper.insert(aes);
        }

    }
