// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.pica.cloud.account.account.server.entity.PermissionRole;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.SaasRoleEnum;
import com.pica.cloud.account.account.server.mapper.PermissionDoctorRoleMapper;
import com.pica.cloud.account.account.server.mapper.PermissionRoleMapper;
import com.pica.cloud.account.account.server.req.HospitalRoleDetailReq;
import com.pica.cloud.account.account.server.req.HospitalRoleListReq;
import com.pica.cloud.account.account.server.resp.*;
import com.pica.cloud.account.account.server.service.HospitalSaasRoleService;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.entity.PicaWarnException;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.foundation.utils.utils.CommonUtil;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @Author qinwh
 * @Date 2022/3/1 19:33
 */
@Service
public class HospitalSaasRoleServiceImpl implements HospitalSaasRoleService {
    @Autowired
    PermissionRoleMapper permissionRoleMapper;
    @Autowired
    PermissionDoctorRoleMapper permissionDoctorRoleMapper;

    @Override
    public int insertAndModify(HospitalRoleDetailReq req, PicaUser user) {
        if (StringUtils.isEmpty(req.getName()) || req.getName().trim().length() > 16) {
            throw new PicaWarnException(PicaResultCode.DATA_EXCEPTION.code(), "姓名长度≤16字符");
        }

        if (StringUtils.isEmpty(req.getRemark()) || req.getName().trim().length() > 100) {
            throw new PicaWarnException(PicaResultCode.DATA_EXCEPTION.code(), "备注长度≤100字符");
        }

        if (Objects.nonNull(req.getId()) && (req.getId() <= SaasRoleEnum.SAAS_DOCTOR.getCode())) {
            throw new PicaWarnException(AccountExceptionEnum.PAT_ACCT_HAS_EXIST.getCode(), "该角色无法编辑或删除");
        }

        int num;
        PermissionRole role = new PermissionRole();
        role.setRoleName(req.getName());
        PermissionRole byNameCode = permissionRoleMapper.selectByNameCode(role);
        if (null != byNameCode) {
            throw new PicaWarnException(AccountExceptionEnum.PAT_ACCT_HAS_EXIST.getCode(), "该角色名称重复");
        }
        if (null != req.getId()) {
            role.setId(req.getId());
            role.setRoleName(req.getName());
            role.setRemark(req.getRemark());
            role.setModifiedId(Long.valueOf(user.getId()));
            role.setModifiedTime(new Date());
            num = permissionRoleMapper.updateByPrimaryKeySelective(role);
        } else {
            role.setRoleCode("saas_admin" + CommonUtil.getRandom(10));
            role.setRemark(req.getRemark());
            role.setCreatedId(Long.valueOf(user.getId()));
            role.setCreatedTime(new Date());
            role.setModifiedId(Long.valueOf(user.getId()));
            role.setModifiedTime(new Date());
            num = permissionRoleMapper.insertSelective(role);
        }
        return num;
    }

    @Override
    public HospitalRoleResp listByPage(HospitalRoleListReq req) {
        List<Integer> list = Arrays.asList(SaasRoleEnum.MAIN_ADMIN_ROLE.getCode(), SaasRoleEnum.ADMIN_ROLE.getCode());
        Map<String, Object> map = new HashMap(4);
        map.put("list", list);
        map.put("name", req.getName());
        int count = permissionRoleMapper.listCountByPage(map);
        if (count < 1) {
            return new HospitalRoleResp(Collections.emptyList(), 0);
        }
        map.put("pageNo", (req.getPageNo() - 1) * req.getPageSize());
        map.put("pageSize", req.getPageSize());
        List<HospitalRoleDto> roleDtos = permissionRoleMapper.listByPage(map);
        roleDtos.stream().forEach(t -> {
            if (Objects.equals(t.getRoleId(), SaasRoleEnum.SAAS_MAIN_ADMIN.getCode())) {
                t.setAdminSaasFlag(1);
            }
        });
        return new HospitalRoleResp(roleDtos, count);
    }

    @Override
    @Transactional
    public int delete(Long id, PicaUser user) {
        if (id == null || id == 0L) {
            throw new PicaWarnException(AccountExceptionEnum.PAT_ACCT_HAS_EXIST.getCode(), "请输入角色id");
        }

        if (id <= SaasRoleEnum.SAAS_DOCTOR.getCode()) {
            throw new PicaWarnException(AccountExceptionEnum.PAT_ACCT_HAS_EXIST.getCode(), "该角色无法删除");
        }

        Map<String, Object> map = new HashMap<>(3);
        map.put("id", id);
        map.put("modifiedId", user.getId());
        map.put("modifiedTime", new Date());
        permissionRoleMapper.deleteById(map);
        permissionDoctorRoleMapper.deleteByRoleId(map);
        return 1;
    }

    @Override
    public HospitalSaasRoleResp querySaasRoles() {
        List<RoleDto> roleDtos = permissionRoleMapper.querySaasRoles();
        return new HospitalSaasRoleResp(roleDtos);
    }

    @Override
    public HospitalRoleDetailDto getRoleDetail(Long id) {
        PermissionRole permissionRole = permissionRoleMapper.selectByPrimaryKey(id);
        HospitalRoleDetailDto dto = new HospitalRoleDetailDto();
        BeanUtils.copyProperties(permissionRole, dto);
        return dto;
    }
}
