// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.pica.cloud.account.account.common.req.AcctPatFamilyDto;
import com.pica.cloud.account.account.common.req.AcctPatFamilyReq;
import com.pica.cloud.account.account.common.req.AcctPatInfoReq;
import com.pica.cloud.account.account.common.req.InitAcctReq;
import com.pica.cloud.account.account.common.resp.PatBindResp;
import com.pica.cloud.account.account.common.resp.PatFamilyResp;
import com.pica.cloud.account.account.server.entity.AcctPatFamily;
import com.pica.cloud.account.account.server.entity.AcctPatInfo;
import com.pica.cloud.account.account.server.entity.AcctPatUnion;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.enums.SourceTypeEnum;
import com.pica.cloud.account.account.server.mapper.AcctPatFamilyMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatInfoMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatUnionMapper;
import com.pica.cloud.account.account.server.service.PatHealthPackService;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.pica.cloud.account.account.server.enums.AccountTypeEnum.union_login_blog;
import static java.util.stream.Collectors.toList;

/**
 * @ClassName PatHealthPackServiceIml
 * @Description TODO
 * @Author peijun.zhao
 * @Date 2020/5/14 18:30
 * @ModifyDate 2020/5/14 18:30
 * @Version 1.0
 */
@Service
public class PatHealthPackServiceIml implements PatHealthPackService {

    @Autowired
    private AcctPatInfoMapper patInfoMapper;

    @Autowired
    private AcctPatUnionMapper patUnionMapper;

    @Autowired
    private AcctPatFamilyMapper patFamilyMapper;

    @Override
    public Integer getAcctIdByUnionId(String unionId) {
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(unionId);
        return patUnion == null ? 0 : patUnion.getAcctId();
    }

    /**
     * @Description 获取居民id是本人的家庭组记录-
     * 作为家庭主账户、本人居民id，单条ok 多条异常数据
     * @Author peijun.zhao
     * @Date 2020/5/14 18:48
     * @ModifyDate 2020/5/14 18:48
     * @Params [patientId]
     * @Return java.lang.Integer
     */
    @Override
    public Integer getAcctIdByPatId(Integer patientId) {
        AcctPatFamily patFamilyAcct = patFamilyMapper.getByPatIdSelf(patientId);
        if (null == patFamilyAcct) {
            //该居民id无 家庭成员记录，返回0
            return 0;
        }
        return patFamilyAcct.getAcctId();
    }

    /**
     * @Description
     * 父亲（监护人）、母亲
     *   儿子、女儿
     *     孙子、孙女、外孙、外孙女
     *
     *   查询：儿子 patId查
     *   返回：父亲，女儿，孙子，孙女 patId
     * @Author peijun.zhao
     * @Date 2020/5/20 17:11
     * @ModifyDate 2020/5/20 17:11
     * @Params [patientId]
     * @Return java.util.List<java.lang.Integer>
     */
    @Override
    public List<Integer> getDirectPatIdsByPatId(Integer patientId) {
        List<Integer> patIds = new ArrayList<>();
        AcctPatFamily member = patFamilyMapper.getByPatIdNotSelf(patientId);
        if (null == member) {
            //该居民id无 家庭成员记录，返回null
            return patIds;
        }

        //add 病人id所在家庭组的监护人- patId
        /*AcctPatFamily parentMember = patFamilyMapper.getSelfByAcctId(memberList.get(0).getAcctId());
        patIds.add(parentMember.getPatientId());*/

        //同级所有成员 包括自身
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(member.getAcctId());
        patIds.addAll(patFamilyList.stream().map(obj -> obj.getPatientId()).collect(toList()));

        //下级所有成员

        //自身监护人
        AcctPatFamily self = patFamilyMapper.getByPatIdSelf(member.getPatientId());
        if(null != self){
            //自己存在监护人身份
            List<AcctPatFamily> sonFamilyList = patFamilyMapper.getListByAcctId(self.getAcctId());
            patIds.addAll(sonFamilyList.stream().map(obj -> obj.getPatientId()).collect(toList()));
        }

        //去除自身patId
        List rtnPatIds = patIds.stream().distinct().filter(obj -> !obj.equals(patientId)).collect(toList());

        return rtnPatIds;
    }

    /**
     * @Description TODO
     * @Author peijun.zhao
     * @Date 2020/5/19 18:03
     * @ModifyDate 2020/5/19 18:03
     * @Params [acctId, containAcct] containAcct：是否包含建立acct的账户
     * @Return com.pica.cloud.account.account.common.resp.PatFamilyResp
     */
    @Override
    public PatFamilyResp getFamilyPatsByAcctId(Integer acctId, Integer containAcct) {
        PatFamilyResp resp = new PatFamilyResp();
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(acctId);
        List<AcctPatFamilyDto> dtos =  new ArrayList<>();
        List<Integer> allPadIds = patFamilyList.stream().map(obj ->obj.getPatientId()).collect(toList());
        if(1 == containAcct){
            resp.setMemberPatIds(allPadIds);
            for(AcctPatFamily family : patFamilyList){
                AcctPatFamilyDto dto = new AcctPatFamilyDto();
                dto.setPatientId(family.getPatientId());
                dto.setRelation(family.getRelation());
                dtos.add(dto);
            }
        }else if(0 == containAcct){
            //不包含建立acct账户的pat数据
            patFamilyList = patFamilyMapper.getListByPatIds(allPadIds);
            for(AcctPatFamily family : patFamilyList){
                AcctPatFamilyDto dto = new AcctPatFamilyDto();
                if(!family.getRelation().equals(1)){
                    dto.setPatientId(family.getPatientId());
                    dto.setRelation(family.getRelation());
                    dtos.add(dto);
                }
            }
            resp.setMemberPatIds(dtos.stream().map(obj ->obj.getPatientId()).filter(x -> x!=null).collect(toList()));
        }

        resp.setMemberList(dtos);
        return resp;
    }

    @Transactional
    @Override
    public Integer saveAcct(AcctPatInfoReq patInfoReq) {
        if(Objects.isNull(patInfoReq.getPatientId()) || patInfoReq.getPatientId() == 0 || Objects.isNull(patInfoReq.getName())
                || Objects.isNull(patInfoReq.getMobile()) ){
            throw new PicaException(AccountExceptionEnum.PAT_SAVE_PARAM_ERROR.getCode(), AccountExceptionEnum.PAT_SAVE_PARAM_ERROR.getMessage());
        }

        String mobileEncrypt = EncryptUtils.encryptContent(patInfoReq.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE);
        //check 手机号是否已是主账户
        AcctPatFamily acctPatFamily = patFamilyMapper.getByPatIdSelf(patInfoReq.getPatientId());
        if (null != acctPatFamily) {
            // 重复时返回已有账户
            return acctPatFamily.getAcctId();
        }

        //手机号无重复时创建
        AcctPatInfo patInfo = new AcctPatInfo();
        patInfo.setName(patInfoReq.getName());
        patInfo.setMobilePhone(mobileEncrypt);
        patInfo.setRegisterProduct(AccountTypeEnum.PRODUCT_TYPE_HEALTH.getCode());
        patInfo.setRegisterSource(SourceTypeEnum.H5.getCode());
        patInfo.setRegTime(new Date());

        patInfo.setDeleteFlag(1);
        patInfo.setCreatedId(0);
        patInfo.setModifiedId(0);
        patInfo.setCreatedTime(new Date());
        patInfo.setModifiedTime(new Date());
        patInfoMapper.insert(patInfo);

        AcctPatFamily patFamily = new AcctPatFamily();
        patFamily.setAcctId(patInfo.getId());
        patFamily.setPatientId(patInfoReq.getPatientId());
        patFamily.setRelation(1);

        patFamily.setDeleteFlag(1);
        patFamily.setCreatedId(0);
        patFamily.setModifiedId(0);
        patFamily.setCreatedTime(new Date());
        patFamily.setModifiedTime(new Date());

        patFamilyMapper.insert(patFamily);
        return patInfo.getId();
    }

    @Transactional
    @Override
    public PatBindResp bindUnion(AcctPatInfoReq patInfoReq) {
        PatBindResp resp = new PatBindResp();
        //unionid是否已被其他账号绑定
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(patInfoReq.getUnionId());
        if (Objects.nonNull(patUnion)) {
            //被绑定
            //且非传入acct_d
            if (!patUnion.getAcctId().equals(patInfoReq.getAcctId())) {
                resp.setBindStatus(false);
                resp.setMsg(AccountExceptionEnum.PAT_UNIONID_BINDED_ERROR.getMessage());
            } else {
                //unionId,已被传入acctId绑定，返回 true
                resp.setBindStatus(true);
            }
        } else {
            //acctId是否已绑定其他unionId
            patUnion = patUnionMapper.selectByAcctId(patInfoReq.getAcctId());
            if (Objects.nonNull(patUnion)) {
                //acctId已有unionId，且不是传入的unionId
                if (!patUnion.getUnionId().equals(patInfoReq.getUnionId())) {
                    resp.setBindStatus(false);
                    resp.setMsg(AccountExceptionEnum.PAT_MOBILE_BINDED_WECHAT_ERROR.getMessage());
                }
            } else {
                //未绑定 新增绑定记录
                AcctPatFamily acct = patFamilyMapper.getSelfByAcctId(patInfoReq.getAcctId());

                AcctPatUnion entity = new AcctPatUnion();
                entity.setAcctId(patInfoReq.getAcctId());
                entity.setUnionId(patInfoReq.getUnionId());
                entity.setUnionType(AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());

                entity.setDeleteFlag(1);
                entity.setCreatedId(acct.getPatientId());
                entity.setModifiedId(acct.getPatientId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());

                patUnionMapper.insert(entity);
                resp.setBindStatus(true);
            }
        }

        return resp;
    }

    @Transactional
    @Override
    public PatFamilyResp saveMember(AcctPatFamilyReq familyReqReq) {
        PatFamilyResp resp = new PatFamilyResp();
        boolean successFlag = true;

        //check member valid
        List<AcctPatFamilyDto> memberList = familyReqReq.getMemberList();
        List<AcctPatFamilyDto> failList = new ArrayList<>();
        List<Integer> patIds = memberList.stream().map(obj -> obj.getPatientId()).collect(toList());
        List<AcctPatFamilyDto> relatedPats = new ArrayList<>();
        List<AcctPatFamily> familyList = patFamilyMapper.getListByPatIds(patIds);
        for(AcctPatFamily member : familyList){
            if(!familyReqReq.getAcctId().equals(member.getAcctId())){
                //居民已被其他账户绑定为家庭成员
                AcctPatFamilyDto failOne = new AcctPatFamilyDto();
                failOne.setPatientId(member.getPatientId());
                failOne.setRelation(member.getRelation());
                failList.add(failOne);
                successFlag = false;
                resp.setFailMsg(AccountExceptionEnum.PAT_MEMBER_HAS_MORE.getMessage());
            }else {
                //居民已被传入acctId关联为家庭成员，不需要再保存
                AcctPatFamilyDto failOneRelated = new AcctPatFamilyDto();
                failOneRelated.setRelation(member.getRelation());
                failOneRelated.setPatientId(member.getPatientId());
                relatedPats.add(failOneRelated);
                successFlag = false;
                resp.setFailMsg(AccountExceptionEnum.PAT_MEMBER_HAS_BINDED_SELF.getMessage());
            }
        }

        //已关联自身居民list
        if(relatedPats.size() > 0){
            successFlag = false;
            failList.addAll(relatedPats);
        }

        //通过后 再建立家庭关系
        if(successFlag){
            AcctPatFamily acct = patFamilyMapper.getSelfByAcctId(familyReqReq.getAcctId());
            for(AcctPatFamilyDto member : memberList){
                AcctPatFamily entity = new AcctPatFamily();
                entity.setAcctId(familyReqReq.getAcctId());
                entity.setPatientId(member.getPatientId());
                entity.setRelation(member.getRelation());

                entity.setDeleteFlag(1);
                entity.setCreatedId(acct.getPatientId());
                entity.setModifiedId(acct.getPatientId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());
                patFamilyMapper.insert(entity);
            }
        }

        resp.setSuccessFlag(successFlag);
        resp.setFailMemberList(failList);
        return resp;
    }

    @Override
    public Integer getFamilyAcctPatId(Integer patId) {
        Integer rtnPatId = patId;
        AcctPatFamily acctMember = patFamilyMapper.getByPatIdSelf(patId);
        if(null != acctMember){
            //本人为监护人 返回自身patId
            rtnPatId = acctMember.getPatientId();
        }else {
            //查询是否存在 作为成员的记录
            AcctPatFamily member = patFamilyMapper.getByPatIdNotSelf(patId);
            if(null != member){
                //本人为成员时记录存在
                AcctPatFamily acctPat = patFamilyMapper.getSelfByAcctId(member.getAcctId());
                rtnPatId = acctPat.getPatientId();
            }else {
                //传入patId不存在家庭记录
            }
        }
        return rtnPatId;
    }

    @Transactional
    @Override
    public Integer initAccts(List<InitAcctReq> list) {
        /*for(InitAcctReq initAcctReq : list){
            initAcctReq.setEncryMobile(EncryptUtils.encryptContent(initAcctReq.getMobile(),EncryptConstants.ENCRYPT_TYPE_MOBILE));
        }*/
        List<Integer> patIds = list.stream().map(obj -> obj.getPatId()).collect(toList());
//        List<String> mobiles = list.stream().map(obj -> obj.getMobile()).collect(toList());
        List<String> encryMobiles = list.stream().map(obj -> obj.getEncryMobile()).collect(toList());
        List<AcctPatFamily> checkPatIdList = patFamilyMapper.getByPatIdSelfList(patIds);
        if(CollectionUtils.isNotEmpty(checkPatIdList)){
            //病人id在成员表中 作为主账户不应该存在
            throw new PicaException(AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getCode(), AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getMessage());
        }
        List<AcctPatInfo> checkMobileList = patInfoMapper.getListByMobiles(encryMobiles);
        if(CollectionUtils.isNotEmpty(checkMobileList)){
            throw new PicaException(AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getCode(), AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getMessage());
        }

        //初始主账户数据
        List<AcctPatInfo> infoList = new ArrayList<>();
        for(InitAcctReq initAcctReq : list){
            AcctPatInfo patInfo = new AcctPatInfo();
            patInfo.setMobilePhone(initAcctReq.getEncryMobile());
            patInfo.setName(initAcctReq.getName());
            infoList.add(patInfo);
        }
        patInfoMapper.initBatch(infoList);

        //获取入库账户ID 关联成员表；
//        List<AcctPatInfo> saveAccts = patInfoMapper.getListByMobiles(encryMobiles);
        List<AcctPatFamily> saveList = new ArrayList<>();
        for(AcctPatInfo acct : infoList){
            for(InitAcctReq req : list){
                AcctPatFamily family = new AcctPatFamily();
                if(req.getEncryMobile().equals(acct.getMobilePhone())){
                    family.setRelation(1);
                    family.setPatientId(req.getPatId());
                    family.setAcctId(acct.getId());
                    saveList.add(family);
                    break;
                }
            }
        }

        patFamilyMapper.initBatch(saveList);
        return 1;
    }
}
