// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.RcRepTypeEnum;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.service.AuthCodeService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.IPUtil;
import com.pica.cloud.account.account.server.util.PICAPDeviceInfo;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.entity.PicaWarnException;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.riskcontrol.riskcontrol.client.CheckCodeClient;
import com.pica.cloud.riskcontrol.riskcontrol.common.req.CheckcodeRiskReq;
import com.pica.cloud.riskcontrol.riskcontrol.common.resp.CheckcodeRiskResp;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * @ClassName AuthCodeServiceImpl
 * @Description
 * @Author Chongwen.jiang
 * @Date 2020/3/5 17:01
 * @ModifyDate 2020/3/5 17:01
 * @Version 1.0
 */
@Service
public class AuthCodeServiceImpl implements AuthCodeService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private CheckCodeClient rcClient;

    @Autowired
    private ICacheClient cacheClient;

    @Autowired
    @Qualifier("accountThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;


    /**
     * @Description 调用风控接口
     * @Author Chongwen.jiang
     * @Date 2020/2/21 17:12
     * @ModifyDate 2020/2/21 17:12
     * @Params [mobile]
     * @Return com.pica.cloud.account.account.server.resp.UnifiedVerificationResp
     */
    @Override
    public void rcValidate(HttpServletRequest request, BaseRequest baseRequest, String deviceInfoStr) {
        //  调用风控接口
        PicaResponse picaResponse = null;
        try {
            PICAPDeviceInfo deviceInfo = JSON.parseObject(deviceInfoStr, PICAPDeviceInfo.class);
            String publicIp = IPUtil.getIpAdrress(request);
            String deviceToken = baseRequest.getDevice_token();
            CheckcodeRiskReq rcReq = new CheckcodeRiskReq();
            rcReq.setMobile(baseRequest.getMobile());
            rcReq.setIp(publicIp);
            rcReq.setDeviceIp(deviceInfo.getDevice_ip());
            rcReq.setDeviceId(deviceToken);
            logger.info("rc-checkcodeRisk-req:{}", JSON.toJSONString(rcReq));
            picaResponse = rcClient.checkcodeRisk(rcReq);
            logger.info("rc-checkcodeRisk-resp:{}", JSON.toJSONString(picaResponse));
        } catch (JSONException e) {
            logger.error("rc-checkcodeRisk-JSONException", e);
        } catch (Exception e) {
            logger.error("rc-checkcodeRisk-invoke-exception", e);
        }

        if (picaResponse != null &&
                PicaResultCode.SUCCESS.code().equals(picaResponse.getCode())) {
            Object data = picaResponse.getData();
            if (Objects.nonNull(data)) {
                CheckcodeRiskResp respData = JSON.parseObject(
                        JSON.toJSONString(data), CheckcodeRiskResp.class);
                if (StringUtils.isNotEmpty(respData.getProcessCode())) {
                    String processCode = respData.getProcessCode();
                    if (StringUtils.isNotEmpty(processCode)) {
                        //  请求返回正常
                        if (processCode.equals(RcRepTypeEnum.RC_SEND_FAIL.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_SEND_FAIL.getCode(),
                                    AccountExceptionEnum.PICA_RC_SEND_FAIL.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_TRIGGER.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_TRIGGER.getCode(),
                                    AccountExceptionEnum.PICA_RC_TRIGGER.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_BLACKLIST.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_BLACKLIST.getCode(),
                                    AccountExceptionEnum.PICA_RC_BLACKLIST.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_DAY_LIMIT.getCode())) {
                            throw new PicaWarnException(AccountExceptionEnum.PICA_RC_DAY_LIMIT.getCode(),
                                    AccountExceptionEnum.PICA_RC_DAY_LIMIT.getMessage());
                        } else if (processCode.equals(RcRepTypeEnum.RC_60_SECOND_LIMIT.getCode())) {
                            //  获取剩余秒数
                            String authCodeKey = Constants.AUTH_CODE_PREFIX + baseRequest.getFlag() + "-" + AESUtil.encryptV0(baseRequest.getMobile()) + "-secure";
                            Long time = cacheClient.get(authCodeKey, Long.class);
                            int remainTime = 59 - (int) (System.currentTimeMillis() - time) / 1000;
                            if (remainTime > 0) {
                                throw new PicaWarnException(AccountExceptionEnum.PICA_RC_60_SECOND_LIMIT.getCode(),
                                        AccountExceptionEnum.PICA_RC_60_SECOND_LIMIT.getMessage().replace("{remainTime}", String.valueOf(remainTime)));
                            }
                        } else {
                            //  processCode=1(允许发送短信验证码)
                        }
                    }
                }
            }
        }

    }

    /**
     * @Description 记录风控数据
     * @Author Chongwen.jiang
     * @Date 2020/2/27 16:53
     * @ModifyDate 2020/2/27 16:53
     * @Params [req, request]
     * @Return void
     */
    @Override
    public void recordRcData(HttpServletRequest request, BaseRequest baseRequest, String deviceIp) {
        CheckcodeRiskReq rec = new CheckcodeRiskReq();
        rec.setMobile(baseRequest.getMobile());
        rec.setIp(IPUtil.getIpAdrress(request));
        rec.setDeviceIp(deviceIp);
        rec.setDeviceId(baseRequest.getDevice_token());
        try {
            threadPoolTaskExecutor.execute(() -> {
                logger.info("recordRcData-execute-{}", JSON.toJSONString(rec));
                PicaResponse resp = rcClient.logSend(rec);
                logger.info("recordRcData-resp-{}", JSON.toJSONString(resp));
            });
        } catch (Exception e) {
            logger.error("recordRcData invoke error", e.getMessage());
        }
    }


}
