// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.entity.*;
import com.pica.cloud.account.account.server.enums.SaasRoleEnum;
import com.pica.cloud.account.account.server.mapper.*;
import com.pica.cloud.account.account.server.req.HospitalSaasUserListReq;
import com.pica.cloud.account.account.server.req.HospitalSaasUserReq;
import com.pica.cloud.account.account.server.resp.*;
import com.pica.cloud.account.account.server.service.AccountService;
import com.pica.cloud.account.account.server.service.HospitalSaasUserService;
import com.pica.cloud.account.account.server.service.PasswordService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.ExecutorServiceUtils;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.foundation.utils.utils.MD5Util;
import com.pica.cloud.trade.store.client.StoreCertifyServiceClient;
import com.pica.cloud.trade.store.resp.certify.StoreCertifyStatus;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author Pica
 * @Date 2022/3/1 15:52
 */
@Service
public class HospitalSaasUserServiceImpl implements HospitalSaasUserService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICacheClient redisClient;
    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;
    @Autowired
    private AccountMapper accountMapper;
    @Autowired
    private PUserRoleMapper pUserRoleMapper;
    @Autowired
    private PermissionDoctorRoleMapper doctorRoleMapper;
    @Autowired
    private HospitalMapper hospitalMapper;

    @Autowired
    private PermissionRoleMapper permissionRoleMapper;

    @Autowired
    private StoreCertifyServiceClient storeCertifyServiceClient;

    @Autowired
    private AccountService accountService;

    @Autowired
    private PasswordService passwordService;

    @Override
    @Transactional
    public int register(HospitalSaasUserReq req, Long doctorId) {
        req.checkInsertParams();
        validatorAdminRole(req.getRoleId(), req.getHospitalId().longValue());
        //得到手机号
        String mobile = req.getMobile();
        //幂等校验
        String nxKey = Constants.REPEAT_REGISTER_PREFIX + mobile;
        Long resultNx = redisClient.setnx(nxKey, mobile);
        if (resultNx == 0) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "重复提交");
        }
        redisClient.expire(nxKey, 2);
        if (resultNx == 1) {
            //手机号加密
            String mobileEncrypt = AESUtil.encryptV0(mobile);
            AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(mobileEncrypt);
            //密码已经存在的处理逻辑
            if (Objects.nonNull(accountInfoEntity) && StringUtils.isNotBlank(accountInfoEntity.getPassword())) {
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "用户已存在，且密码已设置,不可进行编辑");
            }
            //用户不存在的处理逻辑
            if (accountInfoEntity == null) {
                Date currentTime = Calendar.getInstance().getTime();
                String md5Pwd = StringUtils.upperCase(MD5Util.MD5(req.getPassword()));
                // account_info
                AccountInfoEntity accountInfo = buildAccountInfo(mobileEncrypt, currentTime, req.getProductType(), req.getSourceType(), md5Pwd);
                accountInfoDetailMapper.insertSelective(accountInfo);
                Integer acctId = accountInfo.getId();
                accountInfoDetailMapper.updateCreateInfo(acctId);
                // doctor
                Account account = buildDoctorMsg(req, mobileEncrypt, currentTime, req.getSourceType(), md5Pwd, acctId);
                accountMapper.insertSelective(account);
                insertAppRole(account.getId());
                insertSaasRole(req, account.getId(), account.getId());
                redisClient.del(nxKey);
                return 1;
            }

            Account byMobilePhone = accountMapper.getByMobilePhone(mobileEncrypt);
            if (!req.getHospitalId().equals(byMobilePhone.getHospitalId())) {
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "正在添加的用户非本机构下用户");
            }
            Account account = new Account();
            account.setId(byMobilePhone.getId());
            account.setModifyId(doctorId);
            account.setHospitalId(req.getHospitalId());
            account.setModifyTime(Calendar.getInstance().getTime());
            if (StringUtils.isNotEmpty(req.getComment())) {
                account.setComment(req.getComment());
            }
            accountService.updateAccountById(account);
            /** 修改密码 */
            if (StringUtils.isNotEmpty(req.getPassword()) && StringUtils.isEmpty(byMobilePhone.getPassword())) {
                passwordService.modifyPassword(mobileEncrypt, byMobilePhone.getPassword(), StringUtils.upperCase(MD5Util.MD5(req.getPassword())), true);
            }
        }
        return 1;
    }

    private void insertSaasRole(HospitalSaasUserReq req, Long doctorId, Long userId) {
        if (CollectionUtils.isEmpty(req.getRoleId())) {
            return;
        }
        List<PermissionDoctorRole> list = new ArrayList<>();
        req.getRoleId().forEach(t -> {
            PermissionDoctorRole doctorRole = new PermissionDoctorRole();
            doctorRole.setDoctorId(doctorId);
            doctorRole.setHospitalId(req.getHospitalId().longValue());
            doctorRole.setRoleId(t);
            doctorRole.setDeleteFlag(1);
            doctorRole.setCreatedId(userId);
            doctorRole.setCreatedTime(new Date());
            doctorRole.setModifiedId(userId);
            doctorRole.setModifiedTime(new Date());
            list.add(doctorRole);
        });
        doctorRoleMapper.batchInsert(list);
    }

    @Override
    public HospitalSaasUserResp listByPage(HospitalSaasUserListReq req) {
        Map<String, Object> map = new HashMap<>(5);
        map.put("hospitalId", req.getHospitalId());
        map.put("name", req.getName());
        if (StringUtils.isBlank(req.getMobile()) && (Objects.nonNull(req.getMobile()) && req.getMobile().length() >= 1)) {
            map.put("mobile", req.getMobile());
        } else {
            map.put("mobile", AESUtil.encryptV0(req.getMobile()));
        }
        Integer count = accountMapper.listCountByPage(map);
        if (count < 1) {
            return new HospitalSaasUserResp(Collections.emptyList(), 0);
        }
        map.put("pageNo", (req.getPageNo() - 1) * req.getPageSize());
        map.put("pageSize", req.getPageSize());
        List<HospitalSaasUserDto> lists = accountMapper.listByPage(map);
        //手机号 解密脱敏
        lists.stream().forEach(t -> {
            List<RoleDto> roleDtos = doctorRoleMapper.selectRoleByUserId(t.getId());
            t.setRoles(roleDtos);
            String phone = EncryptUtils.decryptContent(t.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY);
            String mixMobile = mixMobile(phone);
            t.setMobile(mixMobile);
            if (req.getLoginFlag() == 0 && Objects.nonNull(t.getRoleId()) && t.getRoleId() < SaasRoleEnum.SAAS_DOCTOR.getCode()) {
                t.setAdminSaasFlag(1);
            }
            if (req.getLoginFlag() == 1 && Objects.nonNull(t.getRoleId()) && t.getRoleId() < SaasRoleEnum.SAAS_MAIN_ADMIN.getCode()) {
                t.setAdminSaasFlag(1);
            }
        });
        return new HospitalSaasUserResp(lists, count);
    }

//    private void validatorAdminRole(Long roleId, Long hospitalId) {
//        int count = accountMapper.adminRoleCountByhospitalId(hospitalId);
//        if (count >= 1 && null != roleId && Objects.equals(roleId, SaasRoleEnum.SAAS_MAIN_ADMIN.getCode())) {
//            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "当前已存在系统管理员,请勿重复设置");
//        }
//    }

    private void validatorAdminRole(List<Long> roleId, Long hospitalId) {
        int count = accountMapper.adminRoleCountByhospitalId(hospitalId);
        if (count >= 1 && CollectionUtils.isNotEmpty(roleId) && roleId.contains(SaasRoleEnum.SAAS_MAIN_ADMIN)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "当前已存在系统管理员,请勿重复设置");
        }
    }

    @Override
    public int upsert(HospitalSaasUserReq req, Long doctorId) {
        if (Objects.isNull(req) || Objects.isNull(req.getHospitalId())) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "参数错误");
        }
        validatorAdminRole(req.getRoleId(), req.getHospitalId().longValue());
        String mobileEncrypt = AESUtil.encryptV0(req.getMobile());
        Account accountExist = accountMapper.getByMobilePhone(mobileEncrypt);
        if (accountExist == null) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "当前用户不存在");
        }
//        HospitalSaasUserEditResp saasUser = accountMapper.selectUserRoleById(req.getId());
//        if (req.getLoginFlag() == 1 && saasUser != null && saasUser.getRoleId() != null && !Objects.equals(req.getRoleId(), saasUser.getRoleId()) && req.getRoleId() != 0 && saasUser.getRoleId() < SaasRoleEnum.SAAS_MAIN_ADMIN.getCode()) {
//            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "当前角色不可编辑");
//        }
//        if (req.getLoginFlag() == 0 && saasUser != null && saasUser.getRoleId() != null && !Objects.equals(req.getRoleId(), saasUser.getRoleId()) && req.getRoleId() != 0 && saasUser.getRoleId() < SaasRoleEnum.SAAS_DOCTOR.getCode()) {
//            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "当前角色不可编辑");
//        }
        Account account = new Account();
        account.setId(req.getId());
        account.setModifyId(doctorId);
        account.setModifyTime(Calendar.getInstance().getTime());
        if (StringUtils.isNotEmpty(req.getComment())) {
            account.setComment(req.getComment());
        }
        accountService.updateAccountById(account);

        /** 修改密码 */
        if (StringUtils.isNotEmpty(req.getPassword()) && StringUtils.isEmpty(accountExist.getPassword())) {
            passwordService.modifyPassword(mobileEncrypt, accountExist.getPassword(), StringUtils.upperCase(MD5Util.MD5(req.getPassword())), true);
        }

        Map<String, Object> map = new HashMap(3);
        map.put("id", req.getId());
        map.put("modifiedId", doctorId);
        map.put("modifiedTime", new Date());
        doctorRoleMapper.deleteByDoctorId(map);
        insertSaasRole(req, req.getId(), doctorId);
        return 1;
    }

    @Override
    @Deprecated
    public int delete(Long id) {
        //删除本身信息,删除人员的权限关联信息,删除人员的协议信息
        // todo
        return 0;
    }

    /**
     * 获取机构用户账户信息
     *
     * @param user
     * @return
     */
    @Override
    public HospitalSaasUserDetailResp userDetail(PicaUser user) {
        HospitalSaasUserDetailResp resp = new HospitalSaasUserDetailResp();
        Account pDoctor = accountMapper.selectById(user.getId());
        if (null == pDoctor) {
            return resp;
        }
        if (null != pDoctor.getHospitalId() && 0L != pDoctor.getHospitalId()) {
            Hospital hospital = hospitalMapper.selectByPrimaryKey(pDoctor.getHospitalId());
            if (null != hospital) {
                resp.setHospitalId(hospital.getId().longValue());
                resp.setHospitalName(hospital.getName());
                resp.setHospitalAddress(hospital.getHospitalAddress());
            }
        } else {
            return resp;
        }
        resp.setName(pDoctor.getName());
        resp.setMobile(EncryptUtils.decryptContent(pDoctor.getMobilePhone(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY));
        // saas角色获取
        Map<String, Long> map = new HashMap<>();
        map.put("doctorId", pDoctor.getId());
        map.put("hospitalId", pDoctor.getHospitalId().longValue());
        List<PermissionRole> roles = permissionRoleMapper.selectByDoctorId(map);
        String roleName = roles.stream().map(t -> t.getRoleName()).distinct().collect(Collectors.joining(","));
        resp.setRoleName(roleName);
        // 获取批发资质
        StoreCertifyStatus storeCertifyStatus = null;
        try {
            storeCertifyStatus = storeCertifyServiceClient.statusByDoctorId(user.getToken(), pDoctor.getId().intValue()).getData();
            resp.setStatusStr(storeCertifyStatus.getCertifyStatusStr());
            resp.setStatus(storeCertifyStatus.getCertifyStatus());
        } catch (Exception e) {
            logger.error("userDetail storeCertifyServiceClient.statusByDoctorId error: {}==doctorId: {}", e, pDoctor.getId());
        }
        return resp;
    }

    @Override
    public HospitalSaasUserEditResp edit(Long id) {
        HospitalSaasUserEditResp resp = accountMapper.selectUserRoleById(id);
        List<RoleDto> roleDtos = doctorRoleMapper.selectRoleByUserId(resp.getId());
        resp.setRoles(roleDtos);
        if (null == resp) {
            return new HospitalSaasUserEditResp();
        }
        resp.setPwdFlag(2);
        //手机号 不为空 解密脱敏
        if (StringUtils.isNotBlank(resp.getMobile())) {
            String phone = EncryptUtils.decryptContent(resp.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY);
            resp.setMobile(phone);
            return resp;
        }
        if (StringUtils.isNotBlank(resp.getPassword())) {
            resp.setPwdFlag(1);
        }
        return resp;
    }

    //处理app端 用户-角色关系
    private void insertAppRole(Long userId) {
        ExecutorServiceUtils.getExecutor().submit(new Runnable() {
            @Override
            public void run() {
                Date date = new Date();
                PUserRole pUserRole = new PUserRole();
                pUserRole.setSystemId(5);
                pUserRole.setUserRoleId(2);
                pUserRole.setUserId(userId.intValue());
                pUserRole.setUserType(1);
                pUserRole.setStatus(2);
                pUserRole.setDeleteFlag(1);
                pUserRole.setCreatId(userId.intValue());
                pUserRole.setCreatTime(date);
                pUserRole.setModifyId(userId.intValue());
                pUserRole.setModifyTime(date);
                pUserRoleMapper.insertSelective(pUserRole);
                pUserRole.setSystemId(1);
                pUserRoleMapper.insertSelective(pUserRole);
                pUserRole.setSystemId(3);
                pUserRoleMapper.insertSelective(pUserRole);
            }
        });
    }

    private AccountInfoEntity buildAccountInfo(String mobileEncrypt, Date currentTime, int productType, int sourceType, String password) {
        AccountInfoEntity accountInfo = new AccountInfoEntity();
        accountInfo.setMobilePhone(mobileEncrypt);
        accountInfo.setPassword(password);
        accountInfo.setCreatedTime(currentTime);
        accountInfo.setCreatedId(0);
        accountInfo.setModifiedId(0);
        accountInfo.setModifiedTime(currentTime);
        accountInfo.setRegTime(currentTime);
        accountInfo.setDeleteFlag(1);
        accountInfo.setSex(0);
        accountInfo.setRegisterProduct(productType);
        accountInfo.setRegisterSource(sourceType);
        accountInfo.setRegTime(currentTime);
        return accountInfo;
    }


    private Account buildDoctorMsg(HospitalSaasUserReq hospitalSaasUserReq, String mobileEncrypt, Date currentTime, int sourceType, String password, Integer acctId) {
        Account account = new Account();
        account.setAcctId(acctId);
        account.setMobilePhone(mobileEncrypt);
        account.setPassword(password);
        if (!StringUtils.isBlank(password)) {
            account.setEntireFlag(3);
        }
        account.setName(hospitalSaasUserReq.getName());
        account.setComment(hospitalSaasUserReq.getComment());
        account.setDeleteFlag(1);
        account.setHospitalId(hospitalSaasUserReq.getHospitalId());
        account.setCreatId(0L);
        account.setModifyId(0L);
        account.setCreatTime(currentTime);
        account.setRegTime(currentTime);
        account.setModifyTime(currentTime);
        account.setFirstLoginTime(currentTime);
        account.setLastLoginTime(currentTime);
        account.setRegisterSource(sourceType);
        return account;
    }


    private static String mixMobile(String mobile) {
        if (org.apache.commons.lang3.StringUtils.isBlank(mobile)) {
            return mobile;
        }
        if (mobile.length() == 11) {
            return (new StringBuilder()).append(mobile.substring(0, 3)).append("****").append(mobile.substring(7)).toString();
        }
        return (new StringBuilder()).append(mobile.substring(0, 1)).append("***").append(mobile.substring(mobile.length())).toString();
    }
}
