// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.controller;

import com.pica.cloud.account.account.common.req.AcctPatFamilyDto;
import com.pica.cloud.account.account.common.req.AcctPatFamilyReq;
import com.pica.cloud.account.account.common.req.AcctPatInfoReq;
import com.pica.cloud.account.account.common.resp.PatBindResp;
import com.pica.cloud.account.account.common.resp.PatFamilyResp;
import com.pica.cloud.account.account.server.entity.AcctPatFamily;
import com.pica.cloud.account.account.server.entity.AcctPatInfo;
import com.pica.cloud.account.account.server.entity.AcctPatUnion;
import com.pica.cloud.account.account.server.mapper.AcctPatFamilyMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatInfoMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatUnionMapper;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaResponse;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static java.util.stream.Collectors.toList;

/**
 * @Description TODO
 * @Author peijun.zhao
 * @Date 2020/5/12 15:55
 * @ModifyDate 2020/5/12 15:55
 * @Params
 * @Return
 */
@Api(description = "健康包")
@RestController
@RequestMapping("/pat-health")
public class PatHealthPackController {

    @Autowired
    private AcctPatInfoMapper patInfoMapper;

    @Autowired
    private AcctPatUnionMapper patUnionMapper;

    @Autowired
    private AcctPatFamilyMapper patFamilyMapper;

    @ApiOperation("获取主账户ID-根据unionId")
    @GetMapping(value = "/acctId/unionId/{unionId}")
    public PicaResponse<Integer> getAcctIdByUnionId(@PathVariable("unionId") String unionId) throws Exception {
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(unionId);
        return PicaResponse.toResponse(patUnion == null ? 0 : patUnion.getAcctId());
    }

    @ApiOperation("获取主账户ID-作为监护人自身patientId")
    @GetMapping(value = "/acctId/patId/{patientId}")
    public PicaResponse<Integer> getAcctIdByPatId(@PathVariable("patientId") Integer patientId) throws Exception {
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getByPatIdSelf(patientId);

        if (CollectionUtils.isEmpty(patFamilyList)) {
            //该居民id无 家庭成员记录，返回0
            return PicaResponse.toResponse(0);
        }
        if (patFamilyList.size() > 1) {
            //db中数据有异常,log todo 抛异常
            return PicaResponse.toResponse(patFamilyList.get(0).getAcctId());
        }
        return PicaResponse.toResponse(patFamilyList.get(0).getAcctId());
    }

    @ApiOperation("上级的监护人 + 下级所有成员 patientId list")
    @GetMapping(value = "/directPatIds/{patientId}")
    public PicaResponse<List<Integer>> getDirectPatIdsByPatId(@PathVariable("patientId") Integer patientId) throws Exception {

        List<Integer> patIds = new ArrayList<>();
        List<AcctPatFamily> memberList = patFamilyMapper.getByPatIdNotSelf(patientId);

        if (CollectionUtils.isEmpty(memberList)) {
            //该居民id无 家庭成员记录，返回null
            return PicaResponse.toResponse(patIds);
        }
        if (memberList.size() > 1) {
            //db中数据有异常,log todo 抛异常
            return PicaResponse.toResponse(memberList.get(0).getAcctId());
        }

        //加入病人id所在家庭组的监护人- 上级
        AcctPatFamily parentMember = patFamilyMapper.getSelfByAcctId(memberList.get(0).getAcctId());
        patIds.add(parentMember.getPatientId());

        //下级所有成员 包括自身
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(memberList.get(0).getAcctId());
        patIds.addAll(patFamilyList.stream().map(obj -> obj.getPatientId()).collect(toList()));
        return PicaResponse.toResponse(patIds);
    }

    @ApiOperation("根据accid获取家庭关系idList")
    @GetMapping(value = "/familyPats/{acctId}")
    public PicaResponse<PatFamilyResp> getFamilyPatsByAcctId(@PathVariable("acctId") Integer acctId) throws Exception {
        PatFamilyResp resp = new PatFamilyResp();
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(acctId);
        resp.setMemberPatIds(patFamilyList.stream().map(obj ->obj.getPatientId()).collect(toList()));
        List<AcctPatFamilyDto> dtos =  new ArrayList<>();
        for(AcctPatFamily family : patFamilyList){
            AcctPatFamilyDto dto = new AcctPatFamilyDto();
            dto.setPatientId(family.getPatientId());
            dto.setRelation(family.getRelation());
            dtos.add(dto);
        }
        resp.setFailMemberList(dtos);
        return PicaResponse.toResponse(resp);
    }

    @ApiOperation("保存主账户")
    @PostMapping(value = "/acct")
    @Transactional
    public PicaResponse<Integer> saveAcct(@RequestBody AcctPatInfoReq patInfoReq) throws Exception {

        if(Objects.isNull(patInfoReq.getPatientId()) || patInfoReq.getPatientId() == 0 || Objects.isNull(patInfoReq.getName())
        || Objects.isNull(patInfoReq.getMobile()) ){
            //log // TODO: 2020/5/14
            return null;
        }

        String mobileEncrypt = EncryptUtils.encryptContent(patInfoReq.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE);
        //check 手机号是否已是主账户
//        AcctPatInfo query = new AcctPatInfo();
//        query.setMobilePhone(mobileEncrypt);
//        List<AcctPatInfo> acctList = patInfoMapper.selectByCondition(query);
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getByPatIdSelf(patInfoReq.getPatientId());
        if (CollectionUtils.isNotEmpty(patFamilyList)) {
            // 重复时返回已有账户
            return PicaResponse.toResponse(patFamilyList.get(0));
        }

        //手机号无重复时创建
        AcctPatInfo patInfo = new AcctPatInfo();
        patInfo.setName(patInfoReq.getName());
        patInfo.setMobilePhone(mobileEncrypt);
        patInfo.setRegisterProduct(1);
        patInfo.setRegisterSource(1); //todo
        patInfo.setRegTime(new Date());

        patInfo.setDeleteFlag(1);
        patInfo.setCreatedId(0); //todo 本人
        patInfo.setModifiedId(0);
        patInfo.setCreatedTime(new Date());
        patInfo.setModifiedTime(new Date());
        patInfoMapper.insert(patInfo);

        AcctPatFamily patFamily = new AcctPatFamily();
        patFamily.setAcctId(patInfo.getId());
        patFamily.setPatientId(patInfoReq.getPatientId());
        patFamily.setRelation(1);

        patFamily.setDeleteFlag(1);
        patFamily.setCreatedId(0); //todo 本人
        patFamily.setModifiedId(0);
        patFamily.setCreatedTime(new Date());
        patFamily.setModifiedTime(new Date());

        patFamilyMapper.insert(patFamily);
        return PicaResponse.toResponse(patInfo.getId());
    }


    @ApiOperation("建立微信与账号关联关系")
    @PostMapping(value = "/acct/bindUnion")
    public PicaResponse<PatBindResp> bindUnion(@RequestBody AcctPatInfoReq patInfoReq) throws Exception {
        PatBindResp resp = new PatBindResp();
        //unionid是否已被其他账号绑定
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(patInfoReq.getUnionId());
        if (Objects.nonNull(patUnion)) {
            //被绑定
            //且非传入acct_d
            if (!patUnion.getAcctId().equals(patInfoReq.getAcctId())) {
                resp.setBindStatus(false);
                resp.setMsg("该unionid已被其他手机号绑定");
            } else {
                //unionId,已被传入acctId绑定，返回 true
                resp.setBindStatus(true);
            }
        } else {
            //acctId是否已绑定其他unionId
            patUnion = patUnionMapper.selectByAcctId(patInfoReq.getAcctId());
            if (Objects.nonNull(patUnion)) {
                //acctId已有unionId，且不是传入的unionId
                if (!patUnion.getUnionId().equals(patInfoReq.getUnionId())) {
                    resp.setBindStatus(false);
                    resp.setMsg("该手机号已绑定其他微信号，请尝试使用其他手机号");
                }
            } else {
                //未绑定 新增绑定记录
                AcctPatUnion entity = new AcctPatUnion();
                entity.setAcctId(patInfoReq.getAcctId());
                entity.setUnionId(patInfoReq.getUnionId());
                entity.setUnionType(1);

                entity.setDeleteFlag(1);
                entity.setCreatedId(0); //todo 本人
                entity.setModifiedId(0);
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());

                patUnionMapper.insert(entity);
                resp.setBindStatus(true);
            }
        }


        return PicaResponse.toResponse(resp);
    }

    @ApiOperation("建立家庭服务关系")
    @PostMapping(value = "/family/member")
    public PicaResponse<PatFamilyResp> saveMember(@RequestBody AcctPatFamilyReq familyReqReq) throws Exception {

        PatFamilyResp resp = new PatFamilyResp();
        boolean successFlag = true;

        //check member valid
        List<AcctPatFamilyDto> memberList = familyReqReq.getMemberList();
        for(AcctPatFamilyDto member : memberList){
            //null check todo
        }

        List<AcctPatFamilyDto> failList = new ArrayList<>();
        List<Integer> patIds = memberList.stream().map(obj -> obj.getPatientId()).collect(toList());
        List<AcctPatFamilyDto> relatedPats = new ArrayList<>();
        List<AcctPatFamily> familyList = patFamilyMapper.getListByPatIds(patIds);
        for(AcctPatFamily member : familyList){
            if(!familyReqReq.getAcctId().equals(member.getAcctId())){
                AcctPatFamilyDto failOne = new AcctPatFamilyDto();
                failOne.setPatientId(member.getPatientId());
                failOne.setRelation(member.getRelation());
                failList.add(failOne);
                successFlag = false;
            }else {
                //居民已被传入acctId关联为家庭成员，不需要再保存
                AcctPatFamilyDto failOneRelated = new AcctPatFamilyDto();
                failOneRelated.setRelation(member.getRelation());
                failOneRelated.setPatientId(member.getPatientId());
                relatedPats.add(failOneRelated);
            }
        }

        //已关联自身居民list
        if(relatedPats.size() > 0){
            successFlag = false;
            failList.addAll(relatedPats);
        }

        //通过后 再建立家庭关系
        if(successFlag){
            for(AcctPatFamilyDto member : memberList){
                AcctPatFamily entity = new AcctPatFamily();
                entity.setAcctId(familyReqReq.getAcctId());
                entity.setPatientId(member.getPatientId());
                entity.setRelation(member.getRelation());

                entity.setDeleteFlag(1);
                entity.setCreatedId(familyReqReq.getAcctId());
                entity.setModifiedId(familyReqReq.getAcctId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());
                patFamilyMapper.insert(entity);
            }
        }

        resp.setSuccessFlag(successFlag);
        resp.setFailMemberList(failList);
        return PicaResponse.toResponse(resp);
    }


    @ApiOperation("根据patientId获取监护人-本人监护人（自身patId），本人成员（监护人patId）")
    @GetMapping(value = "/familyAcct/patId/{patId}")
    public PicaResponse<Integer> getFamilyAcctPatId(@PathVariable("patId") Integer patId) throws Exception {
        Integer rtnPatId = 0;
        List<AcctPatFamily> acctMembers = patFamilyMapper.getByPatIdSelf(patId);
        if(CollectionUtils.isNotEmpty(acctMembers)){
            //本人为监护人 返回自身patId
            rtnPatId = acctMembers.get(0).getPatientId();
        }else {
            //查询是否存在 作为成员的记录
            List<AcctPatFamily> members = patFamilyMapper.getByPatIdNotSelf(patId);
            if(CollectionUtils.isNotEmpty(members)){
                //本人为成员时记录存在
                AcctPatFamily acctPat = patFamilyMapper.getSelfByAcctId(members.get(0).getAcctId());
                rtnPatId = acctPat.getPatientId();
            }else {
                //传入patId不存在家庭记录
            }
        }
        return PicaResponse.toResponse(rtnPatId);
    }




}
