// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.entity.EncryptEntity;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.model.DoctorCardCacheModel;
import com.pica.cloud.account.account.server.req.CardRecognizeEncryptReq;
import com.pica.cloud.account.account.server.req.CardRecognizeReq;
import com.pica.cloud.account.account.server.resp.CardRecognizeResp;
import com.pica.cloud.account.account.server.service.OcrService;
import com.pica.cloud.account.account.server.service.ThreadPoolService;
import com.pica.cloud.account.account.server.util.CryptoUtil;
import com.pica.cloud.base.file.client.UploadOcrClient;
import com.pica.cloud.base.file.common.model.OcrCardDto;
import com.pica.cloud.base.file.common.model.WordsResultDto;
import com.pica.cloud.base.file.common.req.UploadOcrReq;
import com.pica.cloud.base.file.common.resp.UploadOcrResp;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.entity.PicaWarnException;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.foundation.utils.utils.StringUtil;
import com.pica.cloud.riskcontrol.riskcontrol.client.CheckCodeClient;
import com.pica.cloud.riskcontrol.riskcontrol.common.req.CheckcodeRiskReq;
import com.pica.cloud.riskcontrol.riskcontrol.common.resp.CheckcodeRiskResp;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Objects;

/**
 * @ClassName OcrServiceImpl
 * @Description app ocr识别业务实现
 * @Author Chongwen.jiang
 * @Date 2020/4/24 11:57
 * @ModifyDate 2020/4/24 11:57
 * @Version 1.0
 */
@Service
public class OcrServiceImpl implements OcrService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private CheckCodeClient rcClient;

    @Autowired
    private ICacheClient cacheClient;

    @Autowired
    private UploadOcrClient uploadOcrClient;

    @Autowired
    private ThreadPoolService threadPoolService;


    @Override
    public CardRecognizeResp ocrRecognizeApp(PicaUser user, CardRecognizeEncryptReq req, String token,
                                             String ip, Integer sourceType, String deviceIp) {
        CardRecognizeResp resp = new CardRecognizeResp();
        //  数据解密
        logger.info("ocrRecognizeApp-req:{}", JSON.toJSONString(req));
        CardRecognizeReq request;
        try {
            EncryptEntity entity = new EncryptEntity();
            entity.setKey(req.getKey());
            entity.setContent(req.getContent());
            request = CryptoUtil.decrypt(entity, CardRecognizeReq.class);
            request.setFile64Str(req.getFile64Str());
        } catch (Exception e) {
            logger.error("ocrRecognizeApp-request parameters decrypt exception", e);
            throw new PicaWarnException(PicaResultCode.PARAM_IS_INVALID.code(),
                    PicaResultCode.PARAM_IS_INVALID.message());
        }
        logger.info("decrypted req:{}, ip:{}, sourceType:{}, deviceIp:{}",
                JSON.toJSONString(request), ip, sourceType, deviceIp);

        //  入参非空校验
        if (Objects.isNull(request)) {
            logger.error("ocrRecognizeApp-request decrypted object is empty");
            throw new PicaWarnException(PicaResultCode.PARAM_IS_BLANK.code(),
                    PicaResultCode.PARAM_IS_BLANK.message());
        }
        if (StringUtil.isEmpty(request.getName()) ||
                StringUtil.isEmpty(request.getFile64Str())) {
            logger.error("ocrRecognizeApp-request object properties is not complete");
            throw new PicaWarnException(PicaResultCode.PARAM_IS_BLANK.code(),
                    PicaResultCode.PARAM_IS_BLANK.message());
        }

        if (StringUtil.isEmpty(request.getNo())) {
            int needCheck = this.rcValidate(sourceType, ip, user.getMobile(), request.getDevice_token(), deviceIp);
            if (needCheck == -1) {
                //  风控通过，不需要滑块校验
                resp.setNeedCheck(false);

                //  允许直接ocr识别和上传图片
                UploadOcrResp uploadOcrResp = this.invokeOcrAndUpload(request, token);
                //  ocr识别到的数据缓存到redis, 封装响应数据
                this.cacheOcrData(resp, uploadOcrResp);
            } else {
                //  需要风控校验

                String timestamp = String.valueOf(System.currentTimeMillis());
                resp.setNo(timestamp);
                //  缓存标识记录
                String key = Constants.OCR_INVOKED_RC_KEY.replace("{timestamp}", timestamp);
                cacheClient.set(key, 1);
            }
        } else {
            //  校验no有效性
            String key = Constants.OCR_INVOKED_RC_KEY.replace("{timestamp}", request.getNo());
            int cacheNum = cacheClient.incr(key).intValue();
            if (cacheNum == 2) {
                cacheClient.del(key);
                //  调用ocr识别，识别成功后并上传图片
                UploadOcrResp uploadOcrResp = this.invokeOcrAndUpload(request, token);

                //  ocr识别到的数据缓存到redis, 封装响应数据
                this.cacheOcrData(resp, uploadOcrResp);

                //  记录风控数据
                threadPoolService.recordRcData(sourceType, user.getMobile(), ip, request.getDevice_token(), deviceIp);

            } else if (cacheNum == -1) {
                //  非法请求
                logger.info("ocrRecognizeApp timestamp cache not exists");
                throw new PicaWarnException(PicaResultCode.PARAM_IS_INVALID.code(),
                        PicaResultCode.PARAM_IS_INVALID.message());
            } else {
                //  重复提交
                logger.info("ocrRecognizeApp duplicate request");
            }
            resp.setNeedCheck(false);
        }

        return resp;
    }

    /**
     * @Description 风控校验
     * @Author Chongwen.jiang
     * @Date 2020/4/24 14:29
     * @ModifyDate 2020/4/24 14:29
     * @Params [sourceType, publicIp, mobile, deviceToken, deviceIp]
     * @Return int
     */
    private int rcValidate(Integer sourceType, String publicIp, String mobile,
                          String deviceToken, String deviceIp) {
        int needCheck = 1;
        //  调用风控接口
        PicaResponse picaResponse = null;
        try {
            CheckcodeRiskReq rcReq = new CheckcodeRiskReq();
            if (StringUtils.isNotEmpty(publicIp)) {
                rcReq.setIp(publicIp);
            }
            if (StringUtils.isNotEmpty(mobile)) {
                rcReq.setMobile(mobile);
            }
            if (StringUtils.isNotEmpty(deviceToken)) {
                rcReq.setDeviceId(deviceToken);
            }
            if (StringUtils.isNotEmpty(deviceIp)) {
                rcReq.setDeviceIp(deviceIp);
            }
            rcReq.setSourceType(sourceType);
            logger.info("rcValidate-req:{}", rcReq.toString());
            picaResponse = rcClient.checkAuthRisk(rcReq);
            logger.info("rcValidate-{}-resp:{}", mobile, JSON.toJSONString(picaResponse));
        } catch (JSONException e) {
            logger.error("rcValidate-JSONException", e);
        } catch (Exception e) {
            logger.error("rcValidate-invoke-exception", e);
        }

        if (picaResponse != null &&
                PicaResultCode.SUCCESS.code().equals(picaResponse.getCode())) {
            Object data = picaResponse.getData();
            if (Objects.nonNull(data)) {
                CheckcodeRiskResp respData = JSON.parseObject(
                        JSON.toJSONString(data), CheckcodeRiskResp.class);
                if(Objects.nonNull(respData)) {
                    //  1允许直接发送，3需要触发风控
                    if(StringUtil.equals(respData.getProcessCode(), "1")){
                        needCheck = -1;
                    }
                }
            }
        }

        return needCheck;
    }

    /**
     * @Description ocr识别，识别成功后并上传图片
     * @Author Chongwen.jiang
     * @Date 2020/4/24 14:30
     * @ModifyDate 2020/4/24 14:30
     * @Params [file, token]
     * @Return com.pica.cloud.base.file.common.resp.UploadOcrResp
     */
    private UploadOcrResp invokeOcrAndUpload(CardRecognizeReq request, String token) {
        UploadOcrResp ocrData = null;
        try {
            //  调用ocr识别和图片上传
            UploadOcrReq ocrReq = new UploadOcrReq();
            ocrReq.setName(request.getName());
            String originalFilename = request.getName();
            String suffix = originalFilename.substring(originalFilename.lastIndexOf(".") + 1);
            ocrReq.setPostfix(suffix);
            ocrReq.setFile64Str(request.getFile64Str());
            PicaResponse<UploadOcrResp> ocrResp = uploadOcrClient.ocrCard(ocrReq, token);
            logger.info("invokeOcrAndUpload-invoke ocr service response: {}",
                    JSON.toJSONString(ocrResp));
            if (!PicaResultCode.SUCCESS.code().equals(ocrResp.getCode())) {
                logger.error("invokeOcrAndUpload-invoke ocr service response code not success, code: {}",
                        ocrResp.getCode());
                if (Constants.OCR_FAIL.equals(ocrResp.getCode())) {
                    throw new PicaWarnException(AccountExceptionEnum.CARD_FRONT_RECOGNIZE_ERROR.getCode(),
                            AccountExceptionEnum.CARD_FRONT_RECOGNIZE_ERROR.getMessage());
                } else {
                    throw new PicaWarnException(ocrResp.getCode(), ocrResp.getMessage());
                }
            }

            Object data = ocrResp.getData();
            if (Objects.nonNull(data)) {
                UploadOcrResp obj = JSON.parseObject(JSON.toJSONString(data), UploadOcrResp.class);
                if (Objects.nonNull(obj)) {
                    if(!StringUtil.equals(obj.getCode(), "1")) {
                        throw new PicaWarnException(obj.getCode(), obj.getMsg());
                    }
                    //  ocr识别成功，并且上传文件成功
                    ocrData = obj;
                }
            }
        } catch (Exception e) {
            logger.error("invokeOcrAndUpload, upload file exception", e);
            throw new PicaWarnException(PicaResultCode.INTERFACE_INVOKE_ERROR.code(),
                    PicaResultCode.INTERFACE_INVOKE_ERROR.message());
        }

        return ocrData;
    }

    /**
     * @Description ocr识别到的数据缓存到redis,在认证提交时需要使用(api-ws中使用)
     * @Author Chongwen.jiang
     * @Date 2020/4/23 17:30
     * @ModifyDate 2020/4/23 17:30
     * @Params [user, resp, ocrUploadData]
     * @Return void
     */
    private void cacheOcrData(CardRecognizeResp resp, UploadOcrResp ocrUploadData) {
        if (Objects.nonNull(ocrUploadData)) {
            String timeStamp = String.valueOf(System.currentTimeMillis());

            OcrCardDto ocrDto = ocrUploadData.getOcrCardDto();
            if(Objects.nonNull(ocrDto)) {
                WordsResultDto cardDto = ocrDto.getWordsResultDto();

                String name = cardDto.getName();
                String cardNo = cardDto.getCardNo();
                String nation = cardDto.getNation();
                String sex = cardDto.getSex();
                String birthday = cardDto.getBirthDay();
                String nativePlace = cardDto.getNativePlace();

                boolean ocrMistake = StringUtil.isEmpty(name) || StringUtil.isEmpty(cardNo) ||
                        StringUtil.isEmpty(nation) || StringUtil.isEmpty(sex) ||
                        StringUtil.isEmpty(birthday) || StringUtil.isEmpty(nativePlace);
                if (ocrMistake) {
                    logger.error("cacheOcrData-invoke ocr service response value not complete :{}",
                            JSON.toJSONString(ocrDto));
                    throw new PicaWarnException(AccountExceptionEnum.CARD_FRONT_RECOGNIZE_ERROR.getCode(),
                            AccountExceptionEnum.CARD_FRONT_RECOGNIZE_ERROR.getMessage());
                }

                DoctorCardCacheModel dto = new DoctorCardCacheModel(name, sex, nation, birthday, nativePlace, cardNo);
                String cacheKey = Constants.CARD_OCR_INFO_KEY.replace("{timestamp}", timeStamp);
                cacheClient.set(cacheKey, dto, Constants.CARD_OCR_EXPIRE);

                resp.setNo(timeStamp);
                resp.setFileUrl(ocrUploadData.getQiniuPath());
            }
        }
    }


}
