// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.entity.*;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.exception.AccountException;
import com.pica.cloud.account.account.server.log.AccountLogEntityUtils;
import com.pica.cloud.account.account.server.log.AccountLogUtils;
import com.pica.cloud.account.account.server.mapper.*;
import com.pica.cloud.account.account.server.req.HospitalSaasUserListReq;
import com.pica.cloud.account.account.server.req.HospitalSaasUserReq;
import com.pica.cloud.account.account.server.resp.HospitalSaasUserDetailResp;
import com.pica.cloud.account.account.server.resp.HospitalSaasUserDto;
import com.pica.cloud.account.account.server.resp.HospitalSaasUserEditResp;
import com.pica.cloud.account.account.server.resp.HospitalSaasUserResp;
import com.pica.cloud.account.account.server.service.AccountService;
import com.pica.cloud.account.account.server.service.HospitalSaasUserService;
import com.pica.cloud.account.account.server.service.PasswordService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.ExecutorServiceUtils;
import com.pica.cloud.account.account.server.util.captcha.CommonUtils;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.foundation.utils.utils.MD5Util;
import com.pica.cloud.foundation.utils.utils.date.D;
import com.pica.cloud.trade.store.client.StoreCertifyServiceClient;
import com.pica.cloud.trade.store.resp.certify.StoreCertifyStatus;
import org.apache.commons.codec.digest.Md5Crypt;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.ExecutorService;

/**
 * @Author Pica
 * @Date 2022/3/1 15:52
 */
@Service
public class HospitalSaasUserServiceImpl implements HospitalSaasUserService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICacheClient redisClient;
    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;
    @Autowired
    private AccountMapper accountMapper;
    @Autowired
    private PUserRoleMapper pUserRoleMapper;
    @Autowired
    private AgreementLogEntityMapper agreementLogEntityMapper;
    @Autowired
    private PermissionDoctorRoleMapper doctorRoleMapper;
    @Autowired
    private HospitalMapper hospitalMapper;

    @Autowired
    private PermissionRoleMapper permissionRoleMapper;

    @Autowired
    private StoreCertifyServiceClient storeCertifyServiceClient;

    @Autowired
    private AccountService accountService;

    @Autowired
    private PasswordService passwordService;

    @Override
    @Transactional
    public int register(HospitalSaasUserReq req, Long doctorId) {
        req.checkInsertParams();
        //得到手机号
        String mobile = req.getMobile();
        //幂等校验
        String nxKey = Constants.REPEAT_REGISTER_PREFIX + mobile;
        Long resultNx = redisClient.setnx(nxKey, mobile);
        if (resultNx == 0) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "该角色名称重复");
        }
        redisClient.expire(nxKey, 2);
        if (resultNx == 1) {
            //手机号加密
            String mobileEncrypt = AESUtil.encryptV0(mobile);
            AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(mobileEncrypt);
            //密码已经存在的处理逻辑
            if (Objects.nonNull(accountInfoEntity) && StringUtils.isNotBlank(accountInfoEntity.getPassword())) {
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "密码已经存在,不可进行编辑");
            }
            Account byMobilePhone = accountMapper.getByMobilePhone(mobileEncrypt);
            //用户不存在的处理逻辑
            if (accountInfoEntity == null) {
                redisClient.expire(nxKey, 30);
                Date currentTime = Calendar.getInstance().getTime();
                String password = req.getPassword();
                String md5Pwd = MD5Util.MD5(password);
                AccountInfoEntity accountInfo = buildAccountInfo(mobileEncrypt, currentTime, req.getProductType(), req.getSourceType(), md5Pwd);
                accountInfoDetailMapper.insertSelective(accountInfo);
                Integer acctId = accountInfo.getId();
                accountInfoDetailMapper.updateCreateInfo(acctId);
                Account account = buildDoctorMsg(req, mobileEncrypt, currentTime, req.getSourceType(), password, acctId);
                accountMapper.insertSelective(account);
                Long userId = account.getId();
                processAgreement(userId);
                insertAppRole(userId);
                insertSaasRole(req, account.getId(), userId);
                redisClient.del(nxKey);
                return 1;
            }
            //用户存在,但密码不存在的处理逻辑
            if (Objects.nonNull(accountInfoEntity) && StringUtils.isBlank(accountInfoEntity.getPassword())) {
                byMobilePhone.setPassword(req.getPassword());
                byMobilePhone.setComment(req.getComment());
                accountMapper.updateByIdSelective(byMobilePhone);
                if (!Objects.equals(req.getRoleId(), -1)) {
                    insertSaasRole(req, byMobilePhone.getId(), doctorId);
                }
            }
            //用户存在,但密码不存在的处理逻辑
            if (Objects.nonNull(accountInfoEntity) && StringUtils.isBlank(accountInfoEntity.getPassword())) {
                byMobilePhone.setPassword(req.getPassword());
                byMobilePhone.setComment(req.getComment());
                byMobilePhone.setHospitalId(req.getHospitalId());
                accountMapper.updateByIdSelective(byMobilePhone);
                if (!Objects.equals(req.getRoleId(), -1)) {
                    insertSaasRole(req, byMobilePhone.getId(), doctorId);
                }
            }
        }
        return 1;
    }

    private void insertSaasRole(HospitalSaasUserReq req, Long doctorId, Long userId) {
        PermissionDoctorRole doctorRole = new PermissionDoctorRole();
        doctorRole.setDoctorId(doctorId);
        doctorRole.setHospitalId(req.getHospitalId().longValue());
        doctorRole.setRoleId(req.getRoleId());
        doctorRole.setDeleteFlag(1);
        doctorRole.setCreatedId(userId);
        doctorRole.setCreatedTime(new Date());
        doctorRole.setModifiedId(userId);
        doctorRole.setModifiedTime(new Date());
        doctorRoleMapper.insert(doctorRole);
    }


    private void processAgreement(Long userId) {
        ExecutorService executor = ExecutorServiceUtils.getExecutor();
        executor.submit(() -> {
            Integer protocolId = agreementLogEntityMapper.getLatestProtocolId(2);  //获取最新用户协议ID
            PProtocolLog log = new PProtocolLog();
            log.setUserId(userId.toString());
            log.setProtocolId(protocolId);
            log.setUserType(2);
            log.setType(2);
            log.setStatus((short) 1);
            log.setCreatedId(userId.intValue());
            log.setModifiedId(userId.intValue());
            agreementLogEntityMapper.insertProtocolLog(log);
            agreementLogEntityMapper.updateSignNum(protocolId);  //更新用户协议签署数量
            protocolId = agreementLogEntityMapper.getLatestProtocolId(3);  //获取最新隐私协议ID
            log.setProtocolId(protocolId);
            log.setType(3);
            agreementLogEntityMapper.insertProtocolLog(log);
            agreementLogEntityMapper.updateSignNum(protocolId);  //更新隐私协议签署数量
        });
    }

    @Override
    public HospitalSaasUserResp listByPage(HospitalSaasUserListReq req) {
        Map<String, Object> map = new HashMap<>(5);
        map.put("hospitalId", req.getHospitalId());
        map.put("name", req.getName());
        map.put("mobile", req.getMobile());
        Integer count = accountMapper.listCountByPage(map);
        if (count < 1) {
            return new HospitalSaasUserResp(Collections.emptyList(), 0);
        }
        map.put("pageNo", (req.getPageNo() - 1) * req.getPageSize());
        map.put("pageSize", req.getPageSize());

        List<HospitalSaasUserDto> lists = accountMapper.listByPage(map);
        //手机号 解密脱敏
        lists.stream().forEach(t -> {
            String phone = EncryptUtils.decryptContent(t.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY);
            String mixMobile = mixMobile(phone);
            t.setMobile(mixMobile);
        });
        return new HospitalSaasUserResp(lists, count);

    }

    @Override
    public int upsert(HospitalSaasUserReq req, Long doctorId) {
        //走更新的逻辑,只可以更新角色和备注
        String mobileEncrypt = AESUtil.encryptV0(req.getMobile());

        Account accountExist = accountMapper.getByMobilePhone(mobileEncrypt);

        if (accountExist == null) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "当前用户不存在");
        }
        Account account = new Account();
        account.setId(doctorId);
        account.setModifyId(doctorId);
        account.setModifyTime(Calendar.getInstance().getTime());
        if (StringUtils.isNotEmpty(req.getComment())) {
            account.setComment(req.getComment());
        }
        accountService.updateAccountById(account);

        /** 修改密码 */
        if (StringUtils.isNotEmpty(req.getPassword()) && StringUtils.isEmpty(accountExist.getPassword())) {
            passwordService.modifyPassword(mobileEncrypt, accountExist.getPassword(), StringUtils.upperCase(MD5Util.MD5(req.getPassword())), true);
        }

        Map  map =new HashMap(3);
        map.put("id", req.getId());
        map.put("modifiedId", doctorId);
        map.put("modifiedTime", new Date());
        doctorRoleMapper.deleteByDoctorId(map);
        insertSaasRole(req, req.getId(), doctorId);
        return 1;
    }

    @Override
    public int delete(Long id) {
        //删除本身信息,删除人员的权限关联信息,删除人员的协议信息
        // todo
        return 0;
    }

    /**
     * 获取机构用户账户信息
     *
     * @param user
     * @return
     */
    @Override
    public HospitalSaasUserDetailResp userDetail(PicaUser user) {
        HospitalSaasUserDetailResp resp = new HospitalSaasUserDetailResp();
        Account pDoctor = accountMapper.selectById(user.getId());
        if (null == pDoctor) {
            return resp;
        }
        if (null != pDoctor.getHospitalId() && 0L != pDoctor.getHospitalId()) {
            Hospital hospital = hospitalMapper.selectByPrimaryKey(pDoctor.getHospitalId());
            if (null != hospital) {
                resp.setHospitalId(hospital.getId().longValue());
                resp.setHospitalName(hospital.getName());
                resp.setHospitalAddress(hospital.getHospitalAddress());
            } else {
                return resp;
            }
        } else {
            return resp;
        }
        resp.setName(pDoctor.getName());
        resp.setMobile(EncryptUtils.decryptContent(pDoctor.getMobilePhone()
                , EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY));

        // saas角色获取
        Map<String, Long> map = new HashMap<>();
        map.put("doctorId", pDoctor.getId());
        map.put("hospitalId", pDoctor.getHospitalId().longValue());
        List<PermissionRole> roles = permissionRoleMapper.selectByDoctorId(map);
        roles.stream().forEach(o -> {
            resp.setRoleName(resp.getRoleName() + o.getRoleName() + " ");
        });

        // 获取批发资质
        StoreCertifyStatus storeCertifyStatus = null;
        try {
            storeCertifyStatus = storeCertifyServiceClient.statusByDoctorId(user.getToken(), pDoctor.getId().intValue()).getData();
            resp.setStatusStr(storeCertifyStatus.getCertifyStatusStr());
            resp.setStatus(storeCertifyStatus.getCertifyStatus());
        } catch (Exception e) {
            logger.error("userDetail storeCertifyServiceClient.statusByDoctorId error {}", e);
        }

        return resp;
    }

    @Override
    public HospitalSaasUserEditResp edit(Long id) {
        HospitalSaasUserEditResp resp = accountMapper.selectUserRoleById(id);
        if (null == resp) {
            return new HospitalSaasUserEditResp();
        }
        //手机号 解密脱敏
        if (StringUtils.isNotBlank(resp.getMobile())) {
            String phone = EncryptUtils.decryptContent(resp.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY);
            resp.setMobile(phone);
        }
        if (StringUtils.isNotBlank(resp.getPassword())) {
            resp.setPwdFlag(1);
        } else {
            resp.setPwdFlag(2);
        }
        return resp;
    }

    //处理app端 用户-角色关系
    private void insertAppRole(Long userId) {
        ExecutorServiceUtils.getExecutor().submit(new Runnable() {
            @Override
            public void run() {
                Date date = new Date();
                PUserRole pUserRole = new PUserRole();
                pUserRole.setSystemId(5);
                pUserRole.setUserRoleId(2);
                pUserRole.setUserId(userId.intValue());
                pUserRole.setUserType(1);
                pUserRole.setStatus(2);
                pUserRole.setDeleteFlag(1);
                pUserRole.setCreatId(userId.intValue());
                pUserRole.setCreatTime(date);
                pUserRole.setModifyId(userId.intValue());
                pUserRole.setModifyTime(date);
                pUserRoleMapper.insertSelective(pUserRole);
                pUserRole.setSystemId(1);
                pUserRoleMapper.insertSelective(pUserRole);
                pUserRole.setSystemId(3);
                pUserRoleMapper.insertSelective(pUserRole);
            }
        });
    }

    private AccountInfoEntity buildAccountInfo(String mobileEncrypt, Date currentTime, int productType, int sourceType, String password) {
        AccountInfoEntity accountInfo = new AccountInfoEntity();
        accountInfo.setMobilePhone(mobileEncrypt);
        accountInfo.setPassword(password);
        accountInfo.setCreatedTime(currentTime);
        accountInfo.setCreatedId(0);
        accountInfo.setModifiedId(0);
        accountInfo.setModifiedTime(currentTime);
        accountInfo.setRegTime(currentTime);
        accountInfo.setDeleteFlag(1);
        accountInfo.setSex(0);
        accountInfo.setRegisterProduct(productType);
        accountInfo.setRegisterSource(sourceType);
        return accountInfo;
    }


    private Account buildDoctorMsg(HospitalSaasUserReq hospitalSaasUserReq, String mobileEncrypt, Date currentTime, int sourceType, String password, Integer acctId) {
        Account account = new Account();
        account.setAcctId(acctId);
        account.setMobilePhone(mobileEncrypt);
        account.setPassword(password);
        if (!StringUtils.isBlank(password)) {
            account.setEntireFlag(3);
        }
        account.setName(hospitalSaasUserReq.getName());
        account.setComment(hospitalSaasUserReq.getComment());
        account.setDeleteFlag(1);
        account.setHospitalId(hospitalSaasUserReq.getHospitalId());
        account.setCreatId(0L);
        account.setModifyId(0L);
        account.setCreatTime(currentTime);
        account.setModifyTime(currentTime);
        account.setFirstLoginTime(currentTime);
        account.setLastLoginTime(currentTime);
        account.setRegisterSource(sourceType);
        return account;
    }


    private static String mixMobile(String mobile) {
        if (org.apache.commons.lang3.StringUtils.isBlank(mobile)) {
            return mobile;
        }
        if (mobile.length() == 11) {
            return (new StringBuilder()).append(mobile.substring(0, 3)).append("****").append(mobile.substring(7)).toString();
        }
        return (new StringBuilder()).append(mobile.substring(0, 1)).append("***").append(mobile.substring(mobile.length())).toString();
    }
}
