package com.pica.cloud.account.account.server.controller;

import com.pica.cloud.account.account.server.entity.AccountInfoEntity;
import com.pica.cloud.account.account.server.entity.AccountUnionEntity;
import com.pica.cloud.account.account.server.entity.EncryptEntity;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.exception.AccountException;
import com.pica.cloud.account.account.server.mapper.AccountInfoDetailMapper;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.service.AccountUnionService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.AccountUtils;
import com.pica.cloud.account.account.server.util.CryptoUtil;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.CommonUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@Api(description = "短信验证码资源")
@RestController
@RequestMapping(value = "/authCode")
public class SysCodeController extends AccountBaseController {

    private final String AUTH_CODE_PREFIX = "authCode-";

    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;

    @Autowired
    private AccountUnionService accountUnionService;

    @Autowired
    @Qualifier("cacheMigrateClient")
    private ICacheClient cacheClient;

    /**
     * 必须要传递flag类型
     *
     * @param entity
     * @return
     * @throws Exception
     */
    @ApiOperation("获取短信验证码")
    @PostMapping(value = "")
    public PicaResponse getSysCode(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        AccountUtils.checkMobilePhone(request.getMobile());
        processSysCode(request.getMobile(), request.getFlag());
        return PicaResponse.toResponse();
    }

    @ApiOperation("微信获取验证码")
    @PostMapping(value = "/authCode/wechat")
    public PicaResponse getWChatSysCode(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setFlag(AccountTypeEnum.SYSCODE_TYPE_WE_CHAT.getCode());
        AccountUtils.checkMobilePhone(request.getMobile());
        AccountUnionEntity accountUnionEntity = accountUnionService.selectInfoByUnionId(request.getUnionId());
        if (accountUnionEntity != null) {
            processSysCode(request.getMobile(), request.getFlag());
            return PicaResponse.toResponse();
        }
        throw new AccountException(AccountExceptionEnum.PICA_UNBIND_MOBILE);
    }


    /**
     * 验证码发送逻辑
     * 1)随机数生成验证码；
     * 2)验证码失效时间十分钟；
     * 3)同一个业务一个手机号一分钟只能发送一次：提示：请X秒后重试.(用手机号区别用户，用flag区别业务类型)
     *
     * @param mobilePhone
     * @param flag
     */
    private void processSysCode(String mobilePhone, Integer flag) {
        String authCodeKey = this.getAuthCodeKey(mobilePhone, flag.toString());
        String authCodeKeySecure = authCodeKey + "-secure";
        //如果存在，说明刚刚发送过验证码
        if (cacheClient.exists(authCodeKey)) {
            Long time = cacheClient.get(this.getAuthCodeKey(mobilePhone, flag.toString()) + "-secure", Long.class);
            if (time == null) {
                time = 0L;
            }
            int remainTime = 59 - (int) (System.currentTimeMillis() - time) / 1000;
            if (remainTime > 0) {
                throw new AccountException(AccountExceptionEnum.PICA_SYSCODE_RETRY.getCode(),
                        AccountExceptionEnum.PICA_SYSCODE_RETRY.getMessage().replace("X", String.valueOf(remainTime)));
            }
        } else {
            String authCode = CommonUtil.createValidateCode();
            String message = "您的验证码是" + authCode + ",在10分钟内有效。如非本人操作，请忽略本短信！";
            AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(mobilePhone);
            long senderId = accountInfoEntity == null ? 0L : accountInfoEntity.getId();
            cacheClient.set(this.getAuthCodeKey(mobilePhone, flag.toString()), authCode, 600);
            cacheClient.set(authCodeKeySecure, System.currentTimeMillis(), 60);
            super.sendMobileMessage(mobilePhone, message, senderId);
        }
    }

    //获取验证码redis key
    private String getAuthCodeKey(String mobilePhone, String flag) {
        return AUTH_CODE_PREFIX + flag + "-" + AESUtil.encryptV0(mobilePhone);
    }
}
