package com.pica.cloud.account.account.server.controller;

import com.pica.cloud.account.account.server.entity.Account;
import com.pica.cloud.account.account.server.entity.AccountInfoEntity;
import com.pica.cloud.account.account.server.entity.AccountUnionEntity;
import com.pica.cloud.account.account.server.entity.EncryptEntity;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.exception.AccountException;
import com.pica.cloud.account.account.server.mapper.AccountInfoDetailMapper;
import com.pica.cloud.account.account.server.req.BaseRequest;
import com.pica.cloud.account.account.server.service.AccountService;
import com.pica.cloud.account.account.server.service.AccountUnionService;
import com.pica.cloud.account.account.server.service.CaptchaService;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.account.account.server.util.AccountUtils;
import com.pica.cloud.account.account.server.util.CryptoUtil;
import com.pica.cloud.account.account.server.util.RegisterCodeKeyUtils;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.CommonUtil;
import com.pica.cloud.foundation.utils.utils.ValidateUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.logging.Log;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@Api(description = "短信验证码资源")
@RestController
public class AutoCodeController extends AccountBaseController {

    private final String AUTH_CODE_PREFIX = "authCode-";
    private Logger logger = LoggerFactory.getLogger(AccountController.class);

    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;

    @Autowired
    private AccountUnionService accountUnionService;

    @Autowired
    private CaptchaService captchaService;

    @Autowired
    private AccountService accountService;

    @Autowired
    private ICacheClient cacheClient;


    @ApiOperation("获取短信验证码,无需图形验证码,如app端")
    @PostMapping(value = "/authCode")
    public PicaResponse getAuthCode(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        AccountUtils.checkMobilePhone(request.getMobile());
        processSysCode(request.getMobile(), request.getFlag());
        return PicaResponse.toResponse();
    }

    @ApiOperation("获取短信验证码，需要图形验证码,如H5端和PC端；验证码类型 0默认 1注册 2微信登录绑定手机 3修改手机 4重置密码 5忘记密码 7患者招募提交问卷(效验)")
    @PostMapping("/account/authCode")
    public PicaResponse<String> getAuthCodeWithCaptcha(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        String mobilePhone = request.getMobile();
        String flag = request.getFlag() + "";
        String captchaToken = request.getCaptchaToken();
        String captchaAnswer = request.getCaptchaAnswer();
        //校验图形验证码
        if (!captchaService.acknowledge(captchaToken, captchaAnswer)) {
            return PicaResponse.toResponse(null, PicaResultCode.PARAM_IS_INVALID.code(), "图形验证码错误");
        }
        this.checkMobilePhone(mobilePhone);
        String authCode = CommonUtil.createValidateCode();  //随机生成验证码
        String message = "您的验证码是" + authCode + ",在10分钟内有效。如非本人操作，请忽略本短信！";
        //判断账号是否已经存在
        Integer accountIdByMobilePhone = accountService.getAccountIdByMobilePhone(mobilePhone);
        long senderId = accountIdByMobilePhone == null ? 0L : accountIdByMobilePhone;
        //验证码保存到redis，失效时间10分钟
        cacheClient.set(this.getAuthCodeKey(mobilePhone, flag), authCode, 600);
        logger.info(this.getAuthCodeKey(mobilePhone, flag));
        //发送短信
        super.sendMobileMessage(mobilePhone, message, senderId);
        return PicaResponse.toResponse(StringUtils.EMPTY);
    }
    
    @ApiOperation("微信获取验证码")
    @PostMapping(value = "/authCode/wechat")
    public PicaResponse getWChatSysCode(@RequestBody EncryptEntity entity) throws Exception {
        BaseRequest request = CryptoUtil.decrypt(entity, BaseRequest.class);
        request.setFlag(AccountTypeEnum.SYSCODE_TYPE_WE_CHAT.getCode());
        AccountUtils.checkMobilePhone(request.getMobile());
        AccountUnionEntity accountUnionEntity = accountUnionService.selectInfoByUnionId(request.getUnionId());
        if (accountUnionEntity != null) {
            processSysCode(request.getMobile(), request.getFlag());
            return PicaResponse.toResponse();
        }
        throw new AccountException(AccountExceptionEnum.PICA_UNBIND_MOBILE);
    }


    /**
     * 验证码发送逻辑
     * 1)随机数生成验证码；
     * 2)验证码失效时间十分钟；
     * 3)同一个业务一个手机号一分钟只能发送一次：提示：请X秒后重试.(用手机号区别用户，用flag区别业务类型)
     *
     * @param mobilePhone
     * @param flag
     */
    private void processSysCode(String mobilePhone, Integer flag) {
        String authCodeKey = this.getAuthCodeKey(mobilePhone, flag.toString());
        String authCodeKeySecure = authCodeKey + "-secure";
        //如果存在，说明刚刚发送过验证码
        if (cacheClient.exists(authCodeKey)) {
            Long time = cacheClient.get(this.getAuthCodeKey(mobilePhone, flag.toString()) + "-secure", Long.class);
            if (time == null) {
                processSendAuthCode(mobilePhone, flag, authCodeKeySecure);
            } else {
                int remainTime = 59 - (int) (System.currentTimeMillis() - time) / 1000;
                if (remainTime > 0) {
                    throw new AccountException(AccountExceptionEnum.PICA_SYSCODE_RETRY.getCode(),
                            AccountExceptionEnum.PICA_SYSCODE_RETRY.getMessage().replace("X", String.valueOf(remainTime)));
                } else {
                    processSendAuthCode(mobilePhone, flag, authCodeKeySecure);
                }
            }
        } else {
            processSendAuthCode(mobilePhone, flag, authCodeKeySecure);
        }
    }

    /**
     * 调用发送验证码的接口
     *
     * @param mobilePhone
     * @param flag
     * @param authCodeKeySecure
     */
    private void processSendAuthCode(String mobilePhone, Integer flag, String authCodeKeySecure) {
        String authCode = CommonUtil.createValidateCode();
        String message = "您的验证码是" + authCode + ",在10分钟内有效。如非本人操作，请忽略本短信！";
        Integer accountIdByMobilePhone = accountService.getAccountIdByMobilePhone(mobilePhone);
        long senderId = accountIdByMobilePhone == null ? 0L : accountIdByMobilePhone;
        cacheClient.set(this.getAuthCodeKey(mobilePhone, flag.toString()), authCode, 600);
        logger.info("验证码缓存信息----->："+this.getAuthCodeKey(mobilePhone, flag.toString()));
        cacheClient.set(authCodeKeySecure, System.currentTimeMillis(), 60);
        cacheClient.set(RegisterCodeKeyUtils.getRegisterKey(mobilePhone,authCode), mobilePhone, 600);
        super.sendMobileMessage(mobilePhone, message, senderId);
    }

    //获取验证码redis key
    private String getAuthCodeKey(String mobilePhone, String flag) {
        return AUTH_CODE_PREFIX + flag + "-" + AESUtil.encryptV0(mobilePhone);
    }

    //手机格式校验
    private void checkMobilePhone(String mobilePhone) {
        if (StringUtils.isBlank(mobilePhone) || !ValidateUtils.isMobile(mobilePhone)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "请输入正确的手机号");
        }
    }
}
