// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.entity.*;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.enums.SaasRoleEnum;
import com.pica.cloud.account.account.server.mapper.*;
import com.pica.cloud.account.account.common.req.HospitalSaasUserListReq;
import com.pica.cloud.account.account.server.req.HospitalSaasUserReq;
import com.pica.cloud.account.account.server.resp.*;
import com.pica.cloud.account.account.server.service.*;
import com.pica.cloud.account.account.server.util.AESUtil;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.foundation.utils.utils.MD5Util;
import com.pica.cloud.tag.transport.client.ITransportDoctorClient;
import com.pica.cloud.tag.transport.contract.req.StickerProfileDto;
import com.pica.cloud.trade.store.client.StoreCertifyServiceClient;
import com.pica.cloud.trade.store.resp.certify.StoreCertifyStatus;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @Author Pica
 * @Date 2022/3/1 15:52
 */
@Service
public class HospitalSaasUserServiceImpl implements HospitalSaasUserService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICacheClient redisClient;
    @Autowired
    private AccountInfoDetailMapper accountInfoDetailMapper;
    @Autowired
    private AccountMapper accountMapper;
    @Autowired
    private PUserRoleMapper pUserRoleMapper;
    @Autowired
    private PermissionDoctorRoleMapper doctorRoleMapper;
    @Autowired
    private HospitalMapper hospitalMapper;

    @Autowired
    private PermissionRoleMapper permissionRoleMapper;

    @Autowired
    private StoreCertifyServiceClient storeCertifyServiceClient;

    @Autowired
    private AccountService accountService;

    @Autowired
    private PasswordService passwordService;
    @Autowired
    private RegisterService registerService;
    @Autowired
    private ITransportDoctorClient iTransportDoctorClient;

    @Value("${saas.sticker.id}")
    private Integer saasStickerId;

    @Autowired
    private DoctorService doctorService;

    @Override
    @Transactional
    public int register(HospitalSaasUserReq req, PicaUser picaUser) {
        if (req.getRoleId().contains(SaasRoleEnum.SAAS_MAIN_ADMIN.getCode())) {
            PermissionDoctorRole existRole = doctorRoleMapper.selectByHospitalIdRoleId(req.getHospitalId(), SaasRoleEnum.SAAS_MAIN_ADMIN.getCode().intValue());
            if (existRole != null) {
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "saas系统管理员已存在");
            }
        }
        if (CollectionUtils.isNotEmpty(req.getRoleId()) &&
                ((req.getRoleId().contains(SaasRoleEnum.MAIN_ADMIN_ROLE.getCode())) || req.getRoleId().contains(SaasRoleEnum.ADMIN_ROLE.getCode()))) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "主管理员或系统管理员不可被操作");
        }
        if (req.getLoginFlag() == 1 && CollectionUtils.isNotEmpty(req.getRoleId()) &&
                ((req.getRoleId().contains(SaasRoleEnum.MAIN_ADMIN_ROLE.getCode())) || req.getRoleId().contains(SaasRoleEnum.ADMIN_ROLE.getCode()) || req.getRoleId().contains(SaasRoleEnum.SAAS_MAIN_ADMIN.getCode()))) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "saas系统管理员不可被操作");
        }
        if (CollectionUtils.isNotEmpty(req.getRoleId()) &&
                (req.getRoleId().contains(SaasRoleEnum.NULL_ROLE.getCode())) && req.getRoleId().size() > 1) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "角色选择非法");
        }
        //得到手机号
        String mobile = req.getMobile();
        //幂等校验
        String nxKey = Constants.REPEAT_REGISTER_PREFIX + mobile;
        Long resultNx = redisClient.setnx(nxKey, mobile);
        if (resultNx == 0) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "重复提交");
        }
        redisClient.expire(nxKey, 2);
        if (resultNx == 1) {
            //手机号加密
            String mobileEncrypt = AESUtil.encryptV0(mobile);
            AccountInfoEntity accountInfoEntity = accountInfoDetailMapper.selectByMobile(mobileEncrypt);
            Account byMobilePhone = accountMapper.getByMobilePhone(mobileEncrypt);
            if ((null == accountInfoEntity && byMobilePhone != null) || (null != accountInfoEntity && byMobilePhone == null)){
                logger.error("saas注册,account和doctor存在数据不一致情况， req={}", JSON.toJSONString(req));
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "数据有误，本次提交失败");
            }
            //用户不存在的处理逻辑
            if (accountInfoEntity == null) {
                Date currentTime = Calendar.getInstance().getTime();
                String md5Pwd = StringUtils.upperCase(MD5Util.MD5(req.getPassword()));
                // account_info
                AccountInfoEntity accountInfo = buildAccountInfo(mobileEncrypt, currentTime, req.getProductType(), req.getName(),req.getSourceType(), md5Pwd);
                accountInfoDetailMapper.insertSelective(accountInfo);
                Integer acctId = accountInfo.getId();
                accountInfoDetailMapper.updateCreateInfo(acctId);
                // doctor
                Account account = buildDoctorMsg(req, mobileEncrypt, currentTime, req.getSourceType(), md5Pwd, acctId);
                accountMapper.insertSelective(account);
                registerService.processRoleMap(account.getId());
                insertSaasRole(req.getRoleId(), req.getHospitalId().longValue(), account.getId(), account.getId());
                redisClient.del(nxKey);

                //saas用户标签
                CompletableFuture.runAsync(() -> this.refreshTag(account.getId(), picaUser.getToken()));

                return 1;
            }
            //用户存在的处理逻辑 给出提示信息
            Integer hospitalId = byMobilePhone.getHospitalId();
            Hospital hospital = null;
            if (null == hospitalId || null == (hospital = hospitalMapper.selectByPrimaryKey(hospitalId))){
                doctorService.joinHospital(req.getHospitalName(),picaUser.getId(),byMobilePhone.getId().intValue(),req.getHospitalId());
//                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "该手机号已存在，但未加入机构，请至云鹊医App中加入机构");
            }
            if (hospitalId.equals(req.getHospitalId())){
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "该手机号已存在本机构中");
            }
            if (!req.getHospitalId().equals(byMobilePhone.getHospitalId())) {
                throw new PicaException(
                        PicaResultCode.PARAM_IS_INVALID.code(),
                        String.format("该手机号已存在于\"%s\"中，如需修改请至云鹊医App中修改所属机构",hospital.getName()));
            }
        }

        return 1;
    }

    private void insertSaasRole(List<Long> roleIdList, Long hospitalId, Long doctorId, Long userId) {
        if (CollectionUtils.isEmpty(roleIdList)) {
            return;
        }
        List<PermissionDoctorRole> list = new ArrayList<>();
        roleIdList.forEach(t -> {
            PermissionDoctorRole doctorRole = new PermissionDoctorRole();
            doctorRole.setDoctorId(doctorId);
            doctorRole.setHospitalId(hospitalId);
            doctorRole.setRoleId(t);
            doctorRole.setDeleteFlag(1);
            doctorRole.setCreatedId(userId);
            doctorRole.setCreatedTime(new Date());
            doctorRole.setModifiedId(userId);
            doctorRole.setModifiedTime(new Date());
            list.add(doctorRole);
        });
        doctorRoleMapper.batchInsert(list);
    }

    @Override
    public HospitalSaasUserResp listByPage(HospitalSaasUserListReq req) {
        Map<String, Object> map = this.dealParam(req);

        Integer count = accountMapper.listCountByPage(map);
        if (count < 1) {
            return new HospitalSaasUserResp(Collections.emptyList(), 0);
        }
        map.put("pageNo", (req.getPageNo() - 1) * req.getPageSize());
        map.put("pageSize", req.getPageSize());
        List<HospitalSaasUserDto> lists = accountMapper.listByPage(map);
        //手机号 解密脱敏
        lists.stream().forEach(t -> {
            List<RoleDto> roleDtos = doctorRoleMapper.selectRoleByUserId(t.getId());
            t.setRoles(roleDtos);
            String phone = EncryptUtils.decryptContent(t.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY);
            String mixMobile = mixMobile(phone);
            t.setMobile(mixMobile);
        });
        return new HospitalSaasUserResp(lists, count);
    }

    private Map<String, Object> dealParam(HospitalSaasUserListReq req) {
        Map<String, Object> map = new HashMap<>(5);
        map.put("hospitalId", req.getHospitalId());
        map.put("name", req.getName());
        if (StringUtils.isBlank(req.getMobile()) && Objects.nonNull(req.getMobile()) && req.getMobile().length() >= 1) {
            map.put("mobile", req.getMobile());
        } else {
            map.put("mobile", AESUtil.encryptV0(req.getMobile()));
        }
        return map;
    }

    @Override
    public Integer count(HospitalSaasUserListReq req) {
        Map<String, Object> map = this.dealParam(req);
        return accountMapper.listCountByPage(map);
    }

    @Override
    public int upsert(HospitalSaasUserReq req, PicaUser picaUser) {
        if (Objects.isNull(req) || Objects.isNull(req.getHospitalId()) || CollectionUtils.isEmpty(req.getRoleId())) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "参数错误");
        }

        List<Long> inputRoleIdList = new ArrayList<>();
        inputRoleIdList.addAll(req.getRoleId());
        List<Long> delRoleIdList = new ArrayList<>();

        /** 判定saas系统管理员重复 */
        if (inputRoleIdList.contains(SaasRoleEnum.SAAS_MAIN_ADMIN.getCode())) {
            PermissionDoctorRole existRole = doctorRoleMapper.selectByHospitalIdRoleId(req.getHospitalId(), SaasRoleEnum.SAAS_MAIN_ADMIN.getCode().intValue());
            if (existRole != null && !existRole.getDoctorId().equals(req.getId())) {
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "saas系统管理员已存在");
            }
        }

        /** 移除不可编辑的角色 */
        if (req.getLoginFlag() == 0) {
            inputRoleIdList.removeIf(ele -> ele.equals(SaasRoleEnum.MAIN_ADMIN_ROLE.getCode()));
            inputRoleIdList.removeIf(ele -> ele.equals(SaasRoleEnum.ADMIN_ROLE.getCode()));
            delRoleIdList.addAll(Arrays.asList(SaasRoleEnum.MAIN_ADMIN_ROLE.getCode(), SaasRoleEnum.ADMIN_ROLE.getCode()));
        } else if (req.getLoginFlag() == 1) {
            inputRoleIdList.removeIf(ele -> ele.equals(SaasRoleEnum.MAIN_ADMIN_ROLE.getCode()));
            inputRoleIdList.removeIf(ele -> ele.equals(SaasRoleEnum.ADMIN_ROLE.getCode()));
            inputRoleIdList.removeIf(ele -> ele.equals(SaasRoleEnum.SAAS_MAIN_ADMIN.getCode()));
            delRoleIdList.addAll(Arrays.asList(SaasRoleEnum.MAIN_ADMIN_ROLE.getCode(), SaasRoleEnum.ADMIN_ROLE.getCode(), SaasRoleEnum.SAAS_MAIN_ADMIN.getCode()));
        }


        if (CollectionUtils.isNotEmpty(inputRoleIdList)) {
            Map<String, Object> map = new HashMap(3);
            map.put("id", req.getId());
            map.put("delRoleIdList", delRoleIdList);
            map.put("modifiedId", picaUser.getId());
            map.put("modifiedTime", new Date());
            doctorRoleMapper.deleteByDoctorId(map);

            try {
                if (!inputRoleIdList.contains(SaasRoleEnum.NULL_ROLE.getCode())) {
                    insertSaasRole(inputRoleIdList, req.getHospitalId().longValue(), req.getId(), picaUser.getId().longValue());
                } else if (inputRoleIdList.size() > 1) {
                    throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "角色清除与其他角色添加不可以并存");
                }
            } catch (PicaException picaException) {
                throw picaException;
            } finally {
                this.refreshTag(req.getId(), picaUser.getToken());
            }
        }

        /** 修改备注 */
        Account account = new Account();
        account.setId(req.getId());
        account.setModifyId(picaUser.getId().longValue());
        account.setModifyTime(Calendar.getInstance().getTime());
        if (StringUtils.isNotEmpty(req.getComment())) {
            account.setComment(req.getComment());
        }
        accountService.updateAccountById(account);

        /** 修改密码 */
        if (StringUtils.isNotEmpty(req.getPassword())) {
            Account accountExist = accountMapper.selectById(req.getId());
            if (accountExist == null || accountExist.getAcctId() == null) {
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "当前用户不存在");
            }
            if (null == accountExist.getAcctId()){
                throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "数据有误");
            }
            AccountInfoEntity accountInfo = accountInfoDetailMapper.selectByPrimaryKey(accountExist.getAcctId());
            if (accountInfo!= null && StringUtils.isEmpty(accountInfo.getPassword())) {
                passwordService.modifyPassword(accountExist.getMobilePhone(), accountInfo.getPassword(), StringUtils.upperCase(MD5Util.MD5(req.getPassword())), true);
            }
        }

        return 1;
    }

    /**更新用户 saas用户标签**/
    private void refreshTag(Long id, String token) {
        Integer flag = doctorRoleMapper.checkIsSaas(id);
        String valueName = (Objects.nonNull(flag) && flag == 1) ? "是" : "否";
        PicaResponse<Integer> response = iTransportDoctorClient.updateDoctorProfilesById(id.intValue(), Lists.newArrayList(new StickerProfileDto(saasStickerId, valueName)), UUID.randomUUID().toString(), token);
        if (PicaResultCode.SUCCESS.code().equals(response.getCode()) && response.getData() > 0) {
            logger.info("HospitalSaasUserServiceImpl.refreshTag updateDoctorProfilesById success: doctorId-{}", id);
        } else {
            logger.info("HospitalSaasUserServiceImpl.refreshTag updateDoctorProfilesById failed: doctorId-{}", id);
        }
    }

    @Override
    @Deprecated
    public int delete(Long id) {
        //删除本身信息,删除人员的权限关联信息,删除人员的协议信息
        // todo
        return 0;
    }

    /**
     * 获取机构用户账户信息
     *
     * @param user
     * @return
     */
    @Override
    public HospitalSaasUserDetailResp userDetail(PicaUser user) {
        HospitalSaasUserDetailResp resp = new HospitalSaasUserDetailResp();
        Account pDoctor = accountMapper.selectById(user.getId());
        if (null == pDoctor) {
            return resp;
        }
        if (null != pDoctor.getHospitalId() && 0L != pDoctor.getHospitalId()) {
            Hospital hospital = hospitalMapper.selectByPrimaryKey(pDoctor.getHospitalId());
            if (null != hospital) {
                resp.setHospitalId(hospital.getId().longValue());
                resp.setHospitalName(hospital.getName());
                resp.setHospitalAddress(hospital.getHospitalAddress());
            }
        } else {
            return resp;
        }
        resp.setName(pDoctor.getName());
        resp.setMobile(EncryptUtils.decryptContent(pDoctor.getMobilePhone(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY));
        // saas角色获取
        Map<String, Long> map = new HashMap<>();
        map.put("doctorId", pDoctor.getId());
        map.put("hospitalId", pDoctor.getHospitalId().longValue());
        List<PermissionRole> roles = permissionRoleMapper.selectByDoctorId(map);
        String roleName = roles.stream().map(t -> t.getRoleName()).distinct().collect(Collectors.joining(","));
        resp.setRoleName(roleName);
        // 获取批发资质
        StoreCertifyStatus storeCertifyStatus = null;
        try {
            storeCertifyStatus = storeCertifyServiceClient.statusByDoctorId(user.getToken(), pDoctor.getId().intValue()).getData();
            resp.setStatusStr(storeCertifyStatus.getCertifyStatusStr());
            resp.setStatus(storeCertifyStatus.getCertifyStatus());
        } catch (Exception e) {
            logger.error("userDetail storeCertifyServiceClient.statusByDoctorId error: {}==doctorId: {}", e, pDoctor.getId());
        }
        return resp;
    }

    @Override
    public HospitalSaasUserEditResp edit(Long id) {
        HospitalSaasUserEditResp resp = accountMapper.selectUserRoleById(id);
        if (null == resp) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "数据有误");
        }
        List<RoleDto> roleDtos = doctorRoleMapper.selectRoleByUserId(resp.getId());
        resp.setRoles(roleDtos);
        resp.setRoleIds(roleDtos.stream().map(RoleDto::getId).collect(Collectors.toList()));
        resp.setPwdFlag(2);
        AccountInfoEntity accountInfo;
        if (null == resp.getAcctId() || null == (accountInfo=accountInfoDetailMapper.selectByPrimaryKey(resp.getAcctId()))){
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "数据有误");
        }
        if (StringUtils.isNotBlank(accountInfo.getPassword())) {
            resp.setPwdFlag(1);
            resp.setPassword(accountInfo.getPassword());
        }
        //手机号不为空 解密脱敏
        if (StringUtils.isNotBlank(resp.getMobile())) {
            String phone = EncryptUtils.decryptContent(resp.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE, EncryptConstants.ENCRYPT_DECRYPT_KEY);
            resp.setMobile(phone);
        }
        return resp;
    }

    private AccountInfoEntity buildAccountInfo(String mobileEncrypt, Date currentTime, int productType,String name, int sourceType, String password) {
        AccountInfoEntity accountInfo = new AccountInfoEntity();
        accountInfo.setMobilePhone(mobileEncrypt);
        accountInfo.setPassword(password);
        accountInfo.setCreatedTime(currentTime);
        accountInfo.setCreatedId(0);
        accountInfo.setModifiedId(0);
        accountInfo.setName(name);
        accountInfo.setModifiedTime(currentTime);
        accountInfo.setRegTime(currentTime);
        accountInfo.setDeleteFlag(1);
        accountInfo.setSex(0);
        // saas类型目前写死为crm
        accountInfo.setRegisterProduct(AccountTypeEnum.PRODUCT_TYPE_CRM.getCode());
        accountInfo.setRegisterSource(sourceType);
        accountInfo.setRegTime(currentTime);
        return accountInfo;
    }


    private Account buildDoctorMsg(HospitalSaasUserReq hospitalSaasUserReq, String mobileEncrypt, Date currentTime, int sourceType, String password, Integer acctId) {
        Account account = new Account();
        account.setAcctId(acctId);
        account.setMobilePhone(mobileEncrypt);
        account.setPassword(password);
        if (!StringUtils.isBlank(password)) {
            account.setEntireFlag(2);
        }
        account.setName(hospitalSaasUserReq.getName());
        account.setComment(hospitalSaasUserReq.getComment());
        account.setDeleteFlag(1);
        account.setHospitalId(hospitalSaasUserReq.getHospitalId());
        account.setHospital(hospitalSaasUserReq.getHospitalName());
        account.setCreatId(0L);
        account.setModifyId(0L);
        account.setCreatTime(currentTime);
        account.setRegTime(currentTime);
        account.setModifyTime(currentTime);
        account.setFirstLoginTime(currentTime);
        account.setLastLoginTime(currentTime);
        account.setRegisterSource(sourceType);
        return account;
    }


    private static String mixMobile(String mobile) {
        if (org.apache.commons.lang3.StringUtils.isBlank(mobile)) {
            return mobile;
        }
        if (mobile.length() == 11) {
            return (new StringBuilder()).append(mobile.substring(0, 3)).append("****").append(mobile.substring(7)).toString();
        }
        return (new StringBuilder()).append(mobile.substring(0, 1)).append("***").append(mobile.substring(mobile.length())).toString();
    }
}
