package com.pica.cloud.account.account.server.util.captcha;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;

/**
 * @author Laurence Cao
 *
 */
public class CaptchaUtil {

    final static int MAXSIZE = UUID.randomUUID().toString().replace("_", "").length();

    public static String generateUUIDText(int count) {
        int pos = count < MAXSIZE ? count : MAXSIZE;
        pos = pos <= 0 ? 5 : pos;
        return UUID.randomUUID().toString().replace("_", "").substring(0, pos);
    }

    public static byte[] generateImage(String text, CaptchaContext ctx) {
        byte[] ret = null;
        try {
            BufferedImage img = createCaptcha(text, ctx);
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write(img, "png", baos);
            baos.flush();
            ret = baos.toByteArray();
            baos.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return ret;
    }

    public static BufferedImage createCaptcha(String src, CaptchaContext ctx) throws IOException{
        char[] text = src == null ? new char[0] : src.toCharArray();
        if (text == null || text.length == 0) {
            throw new IllegalArgumentException("No captcha text given");
        }

        //BufferedImage image = new BufferedImage(ctx.width, ctx.height, BufferedImage.TYPE_INT_RGB);
        URL url = CaptchaUtil.class.getClassLoader().getResource("1.png");
        BufferedImage image = ImageIO.read(url);
        Graphics2D g2d = image.createGraphics();
        //g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        //g2d.setBackground(Color.WHITE);
        g2d.setColor(Color.BLACK);

        //clearCanvas(g2d, ctx);
        //if (ctx.showGrid) {
        //    drawGrid(g2d, ctx);
        //}

        ctx.height = image.getHeight();
        int charMaxWidth = (int) ((image.getWidth() * 0.8) / text.length);
        int xPos = (int) (image.getWidth() * 0.1);
        for (char ch : text) {
            drawCharacter(g2d, ctx, ch, xPos, charMaxWidth);
            xPos += charMaxWidth;
        }

        g2d.dispose();

        return image;
    }

    /**
     * Clears the canvas.
     */
    private static void clearCanvas(Graphics2D g2d, CaptchaContext ctx) {
        g2d.clearRect(0, 0, ctx.width, ctx.height);
    }

    /**
     * Draws the background grid.
     */
    private static void drawGrid(Graphics2D g2d, CaptchaContext ctx) {
        for (int y = 2; y < ctx.height; y += ctx.gridSize) {
            g2d.drawLine(0, y, ctx.width - 1, y);
        }

        for (int x = 2; x < ctx.width; x += ctx.gridSize) {
            g2d.drawLine(x, 0, x, ctx.height - 1);
        }
    }

    /**
     * Draws a single character.
     *
     * @param g2d {@link Graphics2D} context
     * @param ch character to draw
     * @param x left x position of the character
     * @param boxWidth width of the box
     */
    private static void drawCharacter(Graphics2D g2d, CaptchaContext ctx, char ch, int x, int boxWidth) {
        ThreadLocalRandom rnd = ThreadLocalRandom.current();
        double degree = (rnd.nextDouble() * ctx.rndRotationAmplitude() * 2) - ctx.rndRotationAmplitude();
        double scale = 1 - (rnd.nextDouble() * ctx.rndScaleAmplitude() / 100);

        Graphics2D cg2d = (Graphics2D)g2d.create();
        cg2d.setFont(ctx.font.deriveFont(ctx.rndFontSize()));

        cg2d.translate(x + (boxWidth / 2), ctx.height / 2);
        cg2d.rotate(degree * Math.PI / 90);
        cg2d.scale(scale, scale);

        FontMetrics fm = cg2d.getFontMetrics();
        int charWidth = fm.charWidth(ch);
        int charHeight = fm.getAscent() + fm.getDescent();

        cg2d.drawString(String.valueOf(ch), -(charWidth / 2), fm.getAscent() - (charHeight / 2));

        cg2d.dispose();
    }

}
