package com.pica.cloud.account.account.server.util;

import com.pica.cloud.account.account.server.constants.Constants;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.mapper.DoctorMapper;
import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.foundation.utils.utils.ValidateUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 账户工具类
 */
@Component
public class AccountUtils {
    private Logger logger = LoggerFactory.getLogger(AccountUtils.class);

    @Autowired
    private ICacheClient cacheClient;

    @Autowired
    private DoctorMapper doctorInfoMapper;

    //手机非空和格式校验
    public static void checkMobilePhone(String mobilePhone) {
        if (StringUtils.isBlank(mobilePhone) || !ValidateUtils.isMobile(mobilePhone)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "请输入正确的手机号");
        }
    }

    //密码非空判断
    public static void checkPassword(String password) {
        if (StringUtils.isBlank(password)) {
            throw new PicaException(AccountExceptionEnum.PICA_NOT_EMPTY.getCode(), AccountExceptionEnum.PICA_NOT_EMPTY.getMessage());
        }
    }

    public static boolean checkPasswordRule(String password){
        String regex = "^(?=.*[0-9])(?=.*[a-zA-Z])(.{6,16})$";
        boolean pass = password.matches(regex);
        if(password.length() < 6 || password.length() > 16 || !pass){
            return false;
        }
        return true;
    }

    //获取验证码redis key
    public String getAuthCodeKey(String mobilePhone, String flag) {
        return Constants.AUTH_CODE_PREFIX + flag + "-" + AESUtil.encryptV0(mobilePhone);
    }

    //手机号和验证码校验
    public void checkMobilePhoneAndAuthCode(String mobile, String type, String sysCode) {
        if (StringUtils.isBlank(mobile) || !ValidateUtils.isMobile(mobile)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "请输入正确的手机号");
        }
        String flag = org.apache.commons.lang.StringUtils.isBlank(type) ? "0" : type;
        if (org.apache.commons.lang.StringUtils.isBlank(sysCode)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "短信验证码错误");
        }
        String authCodeKey = getAuthCodeKey(mobile, flag);
        logger.info("验证码缓存信息----->：" + this.getAuthCodeKey(mobile, flag));
        String cacheCode = cacheClient.get(authCodeKey);  //从redis获取验证码
        if (!org.apache.commons.lang.StringUtils.equals(sysCode, cacheCode)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "验证码错误，请重新输入");
        }
        cacheClient.del(authCodeKey);
    }


    public void checkRegisterMobilePhoneAndAuthCode(String mobile, String type, String sysCode) {
        if (StringUtils.isBlank(mobile) || !ValidateUtils.isMobile(mobile)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "请输入正确的手机号");
        }
        if (org.apache.commons.lang.StringUtils.isBlank(sysCode)) {
            throw new PicaException(PicaResultCode.PARAM_IS_INVALID.code(), "短信验证码为空");
        }
        String authCodeKey = RegisterCodeKeyUtils.getRegisterKey(mobile, sysCode);
        Long num = cacheClient.decr(authCodeKey);
        logger.info("key" + authCodeKey);
        logger.info("success" + num);
        if (num == -1) {
            throw new PicaException(PicaResultCode.RESULE_DATA_NONE.code(), "验证码错误，请重新输入");
        }
        if (num > 0 && num < 99) {
            throw new PicaException(PicaResultCode.RESULE_DATA_NONE.code(), "您已注册成功,请直接登录");
        }
        //验证成功以后删除验证码
        cacheClient.del(authCodeKey);

    }

    /**
     * 获取终端来源
     *
     * @param registerSource
     * @return
     */
    public static String getSourceType(Integer registerSource) {
        String sourceType = StringUtils.EMPTY;
        switch (registerSource) {
            case 3:
                sourceType = "web";
                break;
            case 4:
                sourceType = "h5";
                break;
            case 5:
                sourceType = "admin";
                break;
            default:
                sourceType = "app";
        }
        return sourceType;
    }

    /**
     * 通过产品线类型获取用户id
     *
     * @param productType
     * @param AcctId
     * @return
     */
    public Long getUserIdByAcctId(Integer productType, Integer AcctId) {
        return doctorInfoMapper.selectUserIdByAcctId(AcctId);
    }

}
