// Copyright 2016-2101 Pica.
package com.pica.cloud.account.account.server.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.pica.cloud.account.account.common.req.*;
import com.pica.cloud.account.account.common.resp.AcctPatIdResp;
import com.pica.cloud.account.account.common.resp.AcctSelfPatId;
import com.pica.cloud.account.account.common.resp.PatBindResp;
import com.pica.cloud.account.account.common.resp.PatFamilyResp;
import com.pica.cloud.account.account.server.entity.AcctPatFamily;
import com.pica.cloud.account.account.server.entity.AcctPatInfo;
import com.pica.cloud.account.account.server.entity.AcctPatUnion;
import com.pica.cloud.account.account.server.enums.AccountExceptionEnum;
import com.pica.cloud.account.account.server.enums.AccountTypeEnum;
import com.pica.cloud.account.account.server.enums.SourceTypeEnum;
import com.pica.cloud.account.account.server.exception.AccountException;
import com.pica.cloud.account.account.server.mapper.AcctPatFamilyMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatInfoMapper;
import com.pica.cloud.account.account.server.mapper.AcctPatUnionMapper;
import com.pica.cloud.account.account.server.service.PatHealthPackService;
import com.pica.cloud.foundation.encryption.common.constants.EncryptConstants;
import com.pica.cloud.foundation.encryption.util.EncryptUtils;
import com.pica.cloud.foundation.entity.PicaException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

import static java.util.stream.Collectors.toList;

/**
 * @ClassName PatHealthPackServiceIml
 * @Description TODO
 * @Author peijun.zhao
 * @Date 2020/5/14 18:30
 * @ModifyDate 2020/5/14 18:30
 * @Version 1.0
 */
@Service
public class PatHealthPackServiceIml implements PatHealthPackService {

    private Logger logger = LoggerFactory.getLogger(PatHealthPackServiceIml.class);

    @Autowired
    private AcctPatInfoMapper patInfoMapper;

    @Autowired
    private AcctPatUnionMapper patUnionMapper;

    @Autowired
    private AcctPatFamilyMapper patFamilyMapper;

    @Override
    public Integer getAcctIdByUnionId(String unionId) {
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(unionId);
        return patUnion == null ? 0 : patUnion.getAcctId();
    }

    @Override
    public String getUnionIdByAcctId(Integer acctId) {
        AcctPatUnion patUnion = patUnionMapper.selectByAcctId(acctId);
        return patUnion == null ? "" : patUnion.getUnionId();
    }

    /**
     * @Description 获取居民id是本人的家庭组记录-
     * 作为家庭主账户、本人居民id，单条ok 多条异常数据
     * @Author peijun.zhao
     * @Date 2020/5/14 18:48
     * @ModifyDate 2020/5/14 18:48
     * @Params [patientId]
     * @Return java.lang.Integer
     */
    @Override
    public Integer getAcctIdByPatId(Integer patientId) {
        AcctPatFamily patFamilyAcct = patFamilyMapper.getByPatIdSelf(patientId);
        if (null == patFamilyAcct) {
            //该居民id无 家庭成员记录，返回0
            return 0;
        }
        return patFamilyAcct.getAcctId();
    }

    /**
     * @Description 父亲（监护人）、母亲
     * 儿子、女儿---（儿媳 女媳 非家庭成员）
     * 孙子、孙女、外孙、外孙女
     * <p>
     * 儿子（监护人）、儿媳
     * 孙子、孙女
     * <p>
     * 女儿（监护人）、女媳
     * 外孙、外孙女
     * <p>
     * <p>
     * 查询：儿子 patId查
     * 返回：父亲，儿媳，孙子，孙女 patId
     * <p>
     * 查询：父亲
     * 返回：父亲、母亲、儿子、女儿
     * @Author peijun.zhao
     * @Date 2020/5/20 17:11
     * @ModifyDate 2020/5/20 17:11
     * @Params [patientId]
     * @Return java.util.List<java.lang.Integer>
     */
    @Override
    public List<Integer> getDirectPatIdsByPatId(Integer patientId) {
        List<Integer> patIds = new ArrayList<>();
        /*AcctPatFamily member = patFamilyMapper.getByPatIdNotSelf(patientId);
        if (null == member) {
            //该居民id作为成员无 家庭成员记录
            return patIds;
        }*/
        List<Integer> query = new ArrayList<>();
        query.add(patientId);
        List<AcctPatFamily> memberList = patFamilyMapper.getListByPatIds(query);
        if (CollectionUtils.isEmpty(memberList)) {
            //该居民id作为成员无 家庭成员记录
            return patIds;
        }

        //自身监护人
        AcctPatFamily self = patFamilyMapper.getByPatIdSelf(patientId);
        if (null != self) {
            //自己存在监护人身份，家庭关系成员add
            List<AcctPatFamily> sonFamilyList = patFamilyMapper.getListByAcctId(self.getAcctId());
            patIds.addAll(sonFamilyList.stream().map(obj -> obj.getPatientId()).collect(toList()));
        }
        //自身成员
        AcctPatFamily member = patFamilyMapper.getByPatIdNotSelf(patientId);
        if (null != member) {
            //存在自己作为成员身份记录,add 上级监护人
            AcctPatFamily parentAcct = patFamilyMapper.getSelfByAcctId(member.getAcctId());
            if (null != parentAcct) {
                patIds.add(parentAcct.getPatientId());

                //监护人组内所有成员 包括自身
                List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(member.getAcctId());
                patIds.addAll(patFamilyList.stream().map(obj -> obj.getPatientId()).collect(toList()));
            }
        }

        //去除自身patId、去重patId
        List rtnPatIds = patIds.stream().distinct().filter(obj -> !obj.equals(patientId)).collect(toList());

        return rtnPatIds;
    }

    /**
     * @Description TODO
     * @Author peijun.zhao
     * @Date 2020/5/19 18:03
     * @ModifyDate 2020/5/19 18:03
     * @Params [acctId, containAcct] containAcct：是否包含建立acct的账户
     * @Return com.pica.cloud.account.account.common.resp.PatFamilyResp
     */
    @Override
    public PatFamilyResp getFamilyPatsByAcctId(Integer acctId, Integer containAcct) {
        PatFamilyResp resp = new PatFamilyResp();
        List<AcctPatFamily> patFamilyList = patFamilyMapper.getListByAcctId(acctId);
        List<AcctPatFamilyDto> dtos = new ArrayList<>();
        List<Integer> allPadIds = patFamilyList.stream().map(obj -> obj.getPatientId()).collect(toList());
        if (CollectionUtils.isEmpty(allPadIds)) {
            return resp;
        }
        if (1 == containAcct) {
            resp.setMemberPatIds(allPadIds);
            for (AcctPatFamily family : patFamilyList) {
                AcctPatFamilyDto dto = new AcctPatFamilyDto();
                dto.setPatientId(family.getPatientId());
                dto.setRelation(family.getRelation());
                dtos.add(dto);
            }
        } else if (0 == containAcct) {
            //不包含建立acct账户的pat数据
            List<AcctPatFamily> acctList = patFamilyMapper.getByPatIdSelfList(allPadIds);
            List<Integer> acctPatIds = acctList.stream().map(obj -> obj.getPatientId()).collect(toList());
            for (AcctPatFamily family : patFamilyList) {
                if (!acctPatIds.contains(family.getPatientId())) {
                    AcctPatFamilyDto dto = new AcctPatFamilyDto();
                    dto.setPatientId(family.getPatientId());
                    dto.setRelation(family.getRelation());
                    dtos.add(dto);
                }
            }
            resp.setMemberPatIds(dtos.stream().distinct().map(obj -> obj.getPatientId()).filter(x -> x != null).collect(toList()));
        }

        resp.setMemberList(dtos);
        return resp;
    }

    @Transactional
    @Override
    public Integer saveAcct(AcctPatInfoReq patInfoReq) {
        logger.info("saveAcct:{}", JSONObject.toJSONString(patInfoReq));
        if (Objects.isNull(patInfoReq.getPatientId()) || patInfoReq.getPatientId() == 0 || Objects.isNull(patInfoReq.getName())
                || Objects.isNull(patInfoReq.getMobile())) {
            throw new PicaException(AccountExceptionEnum.PAT_SAVE_PARAM_ERROR.getCode(), AccountExceptionEnum.PAT_SAVE_PARAM_ERROR.getMessage());
        }

        String mobileEncrypt = EncryptUtils.encryptContent(patInfoReq.getMobile(), EncryptConstants.ENCRYPT_TYPE_MOBILE);
        //check 手机号是否已是主账户
        AcctPatFamily acctPatFamily = patFamilyMapper.getByPatIdSelf(patInfoReq.getPatientId());
        if (null != acctPatFamily) {
            // 重复时返回已有账户
            return acctPatFamily.getAcctId();
        }

        //手机号无重复时创建
        AcctPatInfo patInfo = new AcctPatInfo();
        patInfo.setName(patInfoReq.getName());
        patInfo.setMobilePhone(mobileEncrypt);
        patInfo.setRegisterProduct(AccountTypeEnum.PRODUCT_TYPE_HEALTH.getCode());
        patInfo.setRegisterSource(SourceTypeEnum.H5.getCode());
        patInfo.setRegTime(new Date());

        patInfo.setDeleteFlag(1);
        patInfo.setCreatedId(0);
        patInfo.setModifiedId(0);
        patInfo.setCreatedTime(new Date());
        patInfo.setModifiedTime(new Date());
        patInfoMapper.insert(patInfo);

        AcctPatFamily patFamily = new AcctPatFamily();
        patFamily.setAcctId(patInfo.getId());
        patFamily.setPatientId(patInfoReq.getPatientId());
        patFamily.setRelation(1);

        patFamily.setDeleteFlag(1);
        patFamily.setCreatedId(0);
        patFamily.setModifiedId(0);
        patFamily.setCreatedTime(new Date());
        patFamily.setModifiedTime(new Date());

        patFamilyMapper.insert(patFamily);
        return patInfo.getId();
    }

    @Transactional
    @Override
    public PatBindResp bindUnion(AcctPatInfoReq patInfoReq) {
        logger.info("bindUnion:{}", JSONObject.toJSONString(patInfoReq));
        PatBindResp resp = new PatBindResp();
        //unionid是否已被其他账号绑定
        AcctPatUnion patUnion = patUnionMapper.selectByUnionId(patInfoReq.getUnionId());
        if (Objects.nonNull(patUnion)) {
            //被绑定
            //且非传入acct_d
            if (!patUnion.getAcctId().equals(patInfoReq.getAcctId())) {
                /** 更新绑定关系 */
                AcctPatUnion acctPatUnion = new AcctPatUnion();
                acctPatUnion.setUnionId(patInfoReq.getUnionId());
                acctPatUnion.setModifiedId(patInfoReq.getAcctId());
                acctPatUnion.setModifiedTime(new Date());
                patUnionMapper.deleteByUnionId(acctPatUnion);

                AcctPatUnion entity = new AcctPatUnion();
                entity.setAcctId(patInfoReq.getAcctId());
                entity.setUnionId(patInfoReq.getUnionId());
                entity.setUnionType(AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());

                entity.setDeleteFlag(1);
                entity.setCreatedId(patInfoReq.getPatientId());
                entity.setModifiedId(patInfoReq.getPatientId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());

                patUnionMapper.insert(entity);
                resp.setBindStatus(true);
//                resp.setMsg(AccountExceptionEnum.PAT_UNIONID_BINDED_ERROR.getMessage());
            } else {
                //unionId,已被传入acctId绑定，返回 true
                resp.setBindStatus(true);
            }
        } else {
            //acctId是否已绑定其他unionId
            patUnion = patUnionMapper.selectByAcctId(patInfoReq.getAcctId());
            if (Objects.nonNull(patUnion)) {
                //acctId已有unionId，且不是传入的unionId
                if (!patUnion.getUnionId().equals(patInfoReq.getUnionId())) {
                    /** 更新绑定关系 */
                    AcctPatUnion acctPatUnion = new AcctPatUnion();
                    acctPatUnion.setUnionId(patUnion.getUnionId());
                    acctPatUnion.setModifiedId(patInfoReq.getAcctId());
                    acctPatUnion.setModifiedTime(new Date());
                    patUnionMapper.deleteByUnionId(acctPatUnion);

                    AcctPatUnion entity = new AcctPatUnion();
                    entity.setAcctId(patInfoReq.getAcctId());
                    entity.setUnionId(patInfoReq.getUnionId());
                    entity.setUnionType(AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());

                    entity.setDeleteFlag(1);
                    entity.setCreatedId(patInfoReq.getPatientId());
                    entity.setModifiedId(patInfoReq.getPatientId());
                    entity.setCreatedTime(new Date());
                    entity.setModifiedTime(new Date());

                    patUnionMapper.insert(entity);
                    resp.setBindStatus(true);
//                    resp.setMsg(AccountExceptionEnum.PAT_MOBILE_BINDED_WECHAT_ERROR.getMessage());
                }
            } else {
                //未绑定 新增绑定记录
                AcctPatFamily acct = patFamilyMapper.getSelfByAcctId(patInfoReq.getAcctId());

                AcctPatUnion entity = new AcctPatUnion();
                entity.setAcctId(patInfoReq.getAcctId());
                entity.setUnionId(patInfoReq.getUnionId());
                entity.setUnionType(AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());

                entity.setDeleteFlag(1);
                entity.setCreatedId(acct.getPatientId());
                entity.setModifiedId(acct.getPatientId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());

                patUnionMapper.insert(entity);
                resp.setBindStatus(true);
            }
        }

        return resp;
    }

    @Transactional
    @Override
    public PatFamilyResp saveMember(AcctPatFamilyReq familyReqReq) {
        logger.info("saveMember:{}", JSONObject.toJSONString(familyReqReq));
        PatFamilyResp resp = new PatFamilyResp();
        boolean successFlag = true;

        //check member valid
        List<AcctPatFamilyDto> memberList = familyReqReq.getMemberList();
        List<AcctPatFamilyDto> failList = new ArrayList<>();
        List<Integer> patIds = memberList.stream().map(obj -> obj.getPatientId()).collect(toList());
        List<AcctPatFamilyDto> relatedPats = new ArrayList<>();
        List<AcctPatFamily> members = patFamilyMapper.getMemberListByPatIds(patIds);
        for (AcctPatFamily member : members) {
            if (!familyReqReq.getAcctId().equals(member.getAcctId())) {
                //居民已被其他账户绑定为家庭成员
                AcctPatFamilyDto failOne = new AcctPatFamilyDto();
                failOne.setPatientId(member.getPatientId());
                failOne.setRelation(member.getRelation());
                failList.add(failOne);
                successFlag = false;
                resp.setFailMsg(AccountExceptionEnum.PAT_MEMBER_HAS_MORE.getMessage());
                resp.setCode(AccountExceptionEnum.PAT_MEMBER_HAS_MORE.getCode());
            } else {
                //居民已被传入acctId关联为家庭成员，不需要再保存
                AcctPatFamilyDto failOneRelated = new AcctPatFamilyDto();
                failOneRelated.setRelation(member.getRelation());
                failOneRelated.setPatientId(member.getPatientId());
                relatedPats.add(failOneRelated);
                successFlag = false;
                resp.setFailMsg(AccountExceptionEnum.PAT_MEMBER_HAS_BINDED_SELF.getMessage());
                resp.setCode(AccountExceptionEnum.PAT_MEMBER_HAS_BINDED_SELF.getCode());
            }
        }

        //已关联自身居民list
        if (relatedPats.size() > 0) {
            successFlag = false;
            failList.addAll(relatedPats);
        }

        //通过后 再建立家庭关系
        if (successFlag) {
            AcctPatFamily acct = patFamilyMapper.getSelfByAcctId(familyReqReq.getAcctId());
            for (AcctPatFamilyDto member : memberList) {
                AcctPatFamily entity = new AcctPatFamily();
                entity.setAcctId(familyReqReq.getAcctId());
                entity.setPatientId(member.getPatientId());
                entity.setRelation(member.getRelation());

                entity.setDeleteFlag(1);
                entity.setCreatedId(acct.getPatientId());
                entity.setModifiedId(acct.getPatientId());
                entity.setCreatedTime(new Date());
                entity.setModifiedTime(new Date());
                patFamilyMapper.insert(entity);
            }
        }

        resp.setSuccessFlag(successFlag);
        resp.setFailMemberList(failList);
        return resp;
    }

    @Override
    public Integer getFamilyAcctPatId(Integer patId) {
        Integer rtnPatId = patId;
        AcctPatFamily acctMember = patFamilyMapper.getByPatIdSelf(patId);
        if (null != acctMember) {
            //本人为监护人 返回自身patId
            rtnPatId = acctMember.getPatientId();
        } else {
            //查询是否存在 作为成员的记录
            AcctPatFamily member = patFamilyMapper.getByPatIdNotSelf(patId);
            if (null != member) {
                //本人为成员时记录存在
                AcctPatFamily acctPat = patFamilyMapper.getSelfByAcctId(member.getAcctId());
                rtnPatId = acctPat.getPatientId();
            } else {
                //传入patId不存在家庭记录
            }
        }
        return rtnPatId;
    }


    /**
     * @Description 根据居民id批量获取对应的主账号对应的patId
     * @Author wangyongbo
     * @Date 2020/6/15 上午11:58
     * @ModifyDate 2020/6/15 上午11:58
     * @Params
     * @Return
     */
    @Override
    public AcctPatIdResp getFamilyAcctPatIds(AcctPatIdReq acctPatIdReq) {
        AcctPatIdResp resp = new AcctPatIdResp();
        Map<Integer, Integer> patFamId = new HashMap<>();

        if (acctPatIdReq == null) {
            throw new AccountException(AccountExceptionEnum.PICA_PARAMS_ERROR);
        }

        /***批量处理*/
        List<Integer> allPatIds = acctPatIdReq.getPatIds();
        int once = 500;
        for (int i = 0; i < allPatIds.size(); i += once) {
            List<Integer> patIds = null;
            if (allPatIds.size() < i + once) {
                patIds = allPatIds.subList(i, allPatIds.size());
            } else {
                patIds = allPatIds.subList(i, i + once);
            }

            /**先获取是主账号本人的居民id*/
            List<AcctPatFamily> acctMembers = patFamilyMapper.getListByPatIdSelfs(patIds);
            for (AcctPatFamily family : acctMembers) {
                patFamId.put(family.getPatientId(), family.getPatientId());
            }

            /** 本人为监护人的patIds*/
            List<Integer> acctMemberPatIds = acctMembers.stream().map(o -> o.getPatientId()).collect(toList());

            /**作为成员的记录*/
            List<Integer> memberIds = patIds.stream().filter(reqPatId -> acctMemberPatIds.stream().noneMatch(patId -> reqPatId.equals(patId))).collect(toList());

            if (CollectionUtils.isNotEmpty(memberIds)) {

                /**获取不是本人的主账号的本人居民id*/
                List<AcctPatFamily> members = patFamilyMapper.getListByPatIdNotSelfs(memberIds);
                if (CollectionUtils.isNotEmpty(members)) {
                    List<Integer> memberAcctIds = members.stream().map(o -> o.getAcctId()).collect(toList());
                    if (CollectionUtils.isNotEmpty(memberAcctIds)) {
                        List<AcctPatFamily> acctPats = patFamilyMapper.getSelfListByAcctIds(memberAcctIds);
                        for (AcctPatFamily family : acctPats) {
                            for (AcctPatFamily member : members) {
                                if (StringUtils.equals(String.valueOf(family.getAcctId()), String.valueOf(member.getAcctId()))) {
                                    patFamId.put(member.getPatientId(), family.getPatientId());
                                }
                            }
                        }
                    }

                    /**没有主账号的patIds*/
                    memberIds.removeAll(members.stream().map(o -> o.getPatientId()).collect(toList()));
                    if (CollectionUtils.isNotEmpty(memberIds)) {
                        for (Integer member : memberIds) {
                            patFamId.put(member, member);
                        }
                    }
                }
            }
        }

        resp.setPatFamId(patFamId);
        return resp;
    }

    @Transactional
    @Override
    public Integer initAccts(List<InitAcctReq> list) {
        /*for(InitAcctReq initAcctReq : list){
            initAcctReq.setEncryMobile(EncryptUtils.encryptContent(initAcctReq.getMobile(),EncryptConstants.ENCRYPT_TYPE_MOBILE));
        }*/
        List<Integer> patIds = list.stream().map(obj -> obj.getPatId()).collect(toList());
//        List<String> mobiles = list.stream().map(obj -> obj.getMobile()).collect(toList());
        List<String> encryMobiles = list.stream().map(obj -> obj.getEncryMobile()).collect(toList());
        List<AcctPatFamily> checkPatIdList = patFamilyMapper.getByPatIdSelfList(patIds);
        if (CollectionUtils.isNotEmpty(checkPatIdList)) {
            //病人id在成员表中 作为主账户不应该存在
            throw new PicaException(AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getCode(), AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getMessage());
        }
        List<AcctPatInfo> checkMobileList = patInfoMapper.getListByMobiles(encryMobiles);
        if (CollectionUtils.isNotEmpty(checkMobileList)) {
            throw new PicaException(AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getCode(), AccountExceptionEnum.PAT_HAS_CREATE_ACCT.getMessage());
        }

        //初始主账户数据
        List<AcctPatInfo> infoList = new ArrayList<>();
        for (InitAcctReq initAcctReq : list) {
            AcctPatInfo patInfo = new AcctPatInfo();
            patInfo.setMobilePhone(initAcctReq.getEncryMobile());
            patInfo.setName(initAcctReq.getName());
            infoList.add(patInfo);
        }
        patInfoMapper.initBatch(infoList);

        //获取入库账户ID 关联成员表；
//        List<AcctPatInfo> saveAccts = patInfoMapper.getListByMobiles(encryMobiles);
        List<AcctPatFamily> saveList = new ArrayList<>();
        for (AcctPatInfo acct : infoList) {
            for (InitAcctReq req : list) {
                AcctPatFamily family = new AcctPatFamily();
                if (req.getEncryMobile().equals(acct.getMobilePhone())) {
                    family.setRelation(1);
                    family.setPatientId(req.getPatId());
                    family.setAcctId(acct.getId());
                    saveList.add(family);
                    break;
                }
            }
        }
        patFamilyMapper.initBatch(saveList);

        //初始 账户联合表
        List<AcctPatUnion> unionList = new ArrayList<>();
        for (AcctPatInfo acct : infoList) {
            for (InitAcctReq req : list) {
                AcctPatUnion patUnion = new AcctPatUnion();
                if (req.getEncryMobile().equals(acct.getMobilePhone())) {
                    if (StringUtils.isNotEmpty(req.getUnionId())) {
                        patUnion.setAcctId(acct.getId());
                        patUnion.setUnionType(AccountTypeEnum.UNION_LOGIN_WE_CHAT.getCode());
                        patUnion.setUnionId(req.getUnionId());
                        unionList.add(patUnion);
                        break;
                    }
                }
            }
        }
        patUnionMapper.initBatch(unionList);

        return 1;
    }

    @Override
    public List<Integer> queryByPage(AcctPatPageReq patPageReq) {
        Integer pageNo = 1;
        Integer pageSize = 100;
        if (null != patPageReq.getPageNo() || 0 != patPageReq.getPageNo()) {
            pageNo = patPageReq.getPageNo();
        }
        if (null != patPageReq.getPageSize() && 0 != patPageReq.getPageSize()) {
            pageSize = patPageReq.getPageSize();
        }

        Map query = new HashMap();
        query.put("begin", (pageNo - 1) * pageSize);
        query.put("end", pageSize);

        return patFamilyMapper.queryPatIdByPage(query);
    }


    @Override
    public AcctSelfPatId getAcctIdByMobile(String mobile) {
        AcctSelfPatId selfPatId = new AcctSelfPatId();
        AcctPatInfo req = new AcctPatInfo();
        req.setMobilePhone(EncryptUtils.encryptContent(mobile, EncryptConstants.ENCRYPT_TYPE_MOBILE));
        AcctPatInfo info = patInfoMapper.selectByCondition(req);
        if (info != null) {
            selfPatId.setAccId(info.getId());
            AcctPatFamily selfPat = patFamilyMapper.getSelfByAcctId(info.getId());
            if (selfPat != null) {
                selfPatId.setSelfPatId(selfPat.getPatientId());
            }
        }
        return selfPatId;
    }

    @Transactional
    public boolean updatePatByAcctId(UpdateAcctPatInfoReq req)
    {
        logger.info("updatePatByAcctId {}", JSON.toJSONString(req));
        //更新 AcctPatInfo 用户名称为 真实姓名
        AcctPatInfo acctPatInfo =new AcctPatInfo();
        acctPatInfo.setId(req.getAcctId());
        acctPatInfo.setName(req.getName());
        acctPatInfo.setModifiedId(req.getNewPatientId());
        acctPatInfo.setModifiedTime(new Date());
        patInfoMapper.updateByPrimaryKeySelective(acctPatInfo);

        //更新 AcctPatFamily 本人的患者id
        AcctPatFamily acctPatFamily = patFamilyMapper.getSelfByAcctId(req.getAcctId());

        AcctPatFamily upAcctPatFamily=new AcctPatFamily();
        upAcctPatFamily.setId(acctPatFamily.getId());
        upAcctPatFamily.setPatientId(req.getNewPatientId());
        upAcctPatFamily.setModifiedId(req.getNewPatientId());
        upAcctPatFamily.setModifiedTime(new Date());
        patFamilyMapper.updateByPrimaryKeySelective(upAcctPatFamily);

        return true;
    }
}
