package com.pica.cloud.online.exam.analysis.server.controller;

import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.RedisClient;
import com.pica.cloud.foundation.utils.entity.PICAUser;
import com.pica.cloud.online.exam.analysis.common.CommonUtils;
import com.pica.cloud.online.exam.analysis.common.constants.CommonConstants;
import com.pica.cloud.online.exam.analysis.common.dto.*;
import com.pica.cloud.online.exam.analysis.common.util.ReturnUtil;
import com.pica.cloud.online.exam.analysis.server.client.IConfigServiceClient;
import com.pica.cloud.online.exam.analysis.server.configuration.PropertiesConfiguration;
import com.pica.cloud.online.exam.analysis.server.entity.*;
import com.pica.cloud.online.exam.analysis.server.service.CHCAnalysisService;
import com.pica.cloud.online.exam.analysis.server.service.DoctorService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.client.loadbalancer.LoadBalancerClient;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author wuminghao
 * @date 2018/8/16 18:10
 */
@RestController
@RequestMapping("/analysis")
@Api(value = "题目分析接口", description = "查看活动及详情，用户点赞、医生回复")
public class AnalysisController {
    @Autowired
    private RedisClient redisClient;

    @Autowired
    private PropertiesConfiguration properties;

    @Autowired
    private LoadBalancerClient slb;

    @Autowired
    private IConfigServiceClient client;

    @Autowired
    private CHCAnalysisService analysisService;

    @Autowired
    private DoctorService doctorService;

    @ApiOperation(value = "获取活动详情", response = PicaResponse.class)
    @RequestMapping(value = "/activityDetail", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<CHCAnalysisDto> getActivityDetail(@RequestParam(required = false) Integer id) {
        PicaResponse.Builder<CHCAnalysisDto> builder = new PicaResponse.Builder<>();

        /**
         * 如果不传活动id进来，活动id就用1
         */
        id = (id == null ? 1 : id);

        try {
            CHCAnalysisDto analysisDto = analysisService.getCHCAnalysisById(id);
            List<AnalysisRoundDto> roundList = analysisService.getRoundListByAnalysisId(id);

            analysisDto.setRoundList(roundList);
            builder.setData(analysisDto);
        } catch (Exception e) {
            e.printStackTrace();
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INNER_ERROR);
        }

        return builder.build();
    }

    @ApiOperation(value = "获取某一轮的题目", response = PicaResponse.class)
    @RequestMapping(value = "/roundDetail/{roundId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<RoundExamTitleDto> getRoundDetail(@PathVariable("roundId") Integer roundId) {
        PicaResponse.Builder<RoundExamTitleDto> builder = new PicaResponse.Builder<>();

        try {
            RoundExamTitleDto roundExamTitleDto = analysisService.getRoundById(roundId);
            List<ExamTitleDto> examTitleDtoList = analysisService.getExamTitleListByRoundId(roundId);

            roundExamTitleDto.setExamTitleList(examTitleDtoList);
            builder.setData(roundExamTitleDto);
        } catch (Exception e) {
            e.printStackTrace();
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INNER_ERROR);
        }

        return builder.build();
    }

    @ApiOperation(value = "获取某一道题目的回复列表", response = PicaResponse.class)
    @RequestMapping(value = "/replyList/{examTitleId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDto> getReplyList(@PathVariable("examTitleId") Integer examTitleId,
                                                        @RequestParam String token) {
        if (token == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_NO_TOKEN);
        }

        PICAUser user = CommonUtils.getUserByToken(redisClient, token);

        if (user == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INVALID_TOKEN);
        }

        PicaResponse.Builder<ExamTitleReplyDto> builder = new PicaResponse.Builder<>();

        try {
            AnalysisRoundExamTitle analysisRoundExamTitle = analysisService.getAnanlysisRoundExamTitleById(examTitleId);

            ExamTitleReplyDto examTitleReplyDto = new ExamTitleReplyDto();
            ExamTitleDto examTitleDto = analysisService.getExamTitleDtoById(examTitleId);
            List<ReplyDto> replyDtoList = analysisService.getReplyListDtoByExamTitleId(examTitleId, user.getId());

            examTitleReplyDto.setAnalysisId(analysisRoundExamTitle.getAnalysisId());
            examTitleReplyDto.setRoundId(analysisRoundExamTitle.getRoundId());
            examTitleReplyDto.setExamTitle(examTitleDto);
            examTitleReplyDto.setReplyList(replyDtoList);

            builder.setData(examTitleReplyDto);
        } catch (Exception e) {
            e.printStackTrace();
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INNER_ERROR);
        }

        return builder.build();
    }

    @ApiOperation(value = "查看回复详情", response = PicaResponse.class)
    @RequestMapping(value = "/replyDetail/{replyId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDetailDto> getReplyDetail(@PathVariable("replyId") Integer replyId,
                                                                @RequestParam String token) {
        if (token == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_NO_TOKEN);
        }

        PICAUser user = CommonUtils.getUserByToken(redisClient, token);

        if (user == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INVALID_TOKEN);
        }

        PicaResponse.Builder<ExamTitleReplyDetailDto> builder = new PicaResponse.Builder<>();

        try {
            ExamTitleReplyDetailDto examTitleReplyDetailDto = new ExamTitleReplyDetailDto();

            Reply reply =  analysisService.getReplyById(replyId);
            AnalysisRoundExamTitle analysisRoundExamTitle = analysisService.getAnanlysisRoundExamTitleById(reply.getAnalysisRoundExamTitleId());

            ExamTitleDto examTitleDto = analysisService.getExamTitleDtoById(reply.getAnalysisRoundExamTitleId());
            ReplyDto replyDto = analysisService.getReplyDtoByReply(reply, user.getId());

            examTitleReplyDetailDto.setAnalysisId(analysisRoundExamTitle.getAnalysisId());
            examTitleReplyDetailDto.setRoundId(analysisRoundExamTitle.getRoundId());
            examTitleReplyDetailDto.setExamTitle(examTitleDto);
            examTitleReplyDetailDto.setReply(replyDto);

            builder.setData(examTitleReplyDetailDto);
        } catch (Exception e) {
            e.printStackTrace();
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INNER_ERROR);
        }

        return builder.build();
    }

    @ApiOperation(value = "点赞", response = PicaResponse.class)
    @RequestMapping(value = "/starReply/{replyId}/analysis/{analysisId}/round/{roundId}/examTitle/{examTitleId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse starReply(@PathVariable("analysisId") Integer analysisId, @PathVariable("roundId") Integer roundId,
                                  @PathVariable("examTitleId") Integer examTitleId, @PathVariable("replyId") Integer replyId,
                                  @RequestParam String token) {
        if (token == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_NO_TOKEN);
        }

        PICAUser user = CommonUtils.getUserByToken(redisClient, token);

        if (user == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INVALID_TOKEN);
        }

        Doctor doctor = doctorService.getDoctorById(user.getId());

        /**
         * 未认证用户直接返回
         */
        if (!doctor.getStatus().equals(CommonConstants.DOCTOR_STATUS_3)) {
            return ReturnUtil.getPicaResponse(PicaResultCode.INTERFACE_FORBID_VISIT);
        }

        if (0 != analysisService.insertStarRecord(analysisId, roundId, examTitleId, replyId, user.getId())) {
            /**
             * 已经点过赞
             */
            return ReturnUtil.getPicaResponse(PicaResultCode.DATA_ALREADY_EXISTED);
        }
        return ReturnUtil.getPicaResponse(PicaResultCode.SUCCESS);
    }

    @ApiOperation(value = "取消点赞", response = PicaResponse.class)
    @RequestMapping(value = "/cancelStar/{replyId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse cancelStar( @PathVariable("replyId") Integer replyId, @RequestParam String token) {
        if (token == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_NO_TOKEN);
        }

        PICAUser user = CommonUtils.getUserByToken(redisClient, token);

        if (user == null) {
            return ReturnUtil.getPicaResponse(PicaResultCode.SYSTEM_INVALID_TOKEN);
        }

        if (0 != analysisService.removeStarRecord(replyId, user.getId())) {
            return ReturnUtil.getPicaResponse(PicaResultCode.RESULE_DATA_NONE);
        }

        return ReturnUtil.getPicaResponse(PicaResultCode.SUCCESS);
    }
}
