package com.pica.cloud.online.exam.analysis.server.service.impl;

import com.pica.cloud.online.exam.analysis.common.CommonUtils;
import com.pica.cloud.online.exam.analysis.common.constants.CommonConstants;
import com.pica.cloud.online.exam.analysis.common.dto.*;
import com.pica.cloud.online.exam.analysis.server.entity.*;
import com.pica.cloud.online.exam.analysis.server.mapper.*;
import com.pica.cloud.online.exam.analysis.server.service.CHCAnalysisService;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author wuminghao
 * @date 2018/8/17 9:45
 */
@Service
public class CHCAnalysisServiceImpl implements CHCAnalysisService {
    static final String[] OPTIONS = {"A", "B", "C", "D", "E", "F", "G", "H"};

    @Autowired
    CHCAnalysisMapper analysisMapper;

    @Autowired
    AnalysisRoundMapper analysisRoundMapper;

    @Autowired
    AnalysisRoundExamTitleMapper analysisRoundExamTitleMapper;

    @Autowired
    StarRecordMapper starRecordMapper;

    @Autowired
    ReplyMapper replyMapper;

    @Autowired
    DoctorMapper doctorMapper;


    @Override
    public CHCAnalysisDto getCHCAnalysisById(Integer analysisId) {

        CHCAnalysis analysis = analysisMapper.selectByPrimaryKey(analysisId);
        if (null ==  analysis) {
            return null;
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        CHCAnalysisDto analysisDto = new CHCAnalysisDto();
        analysisDto.setId(analysis.getId());
        analysisDto.setName(analysis.getAnalysisName());
        analysisDto.setMainTitle(analysis.getMainTitle());
        analysisDto.setSubTitle(analysis.getSubTitle());
        analysisDto.setIntro(analysis.getIntro());
        analysisDto.setStartTime(sdf.format(analysis.getStartTime()));
        analysisDto.setEndTime(sdf.format(analysis.getEndTime()));
        analysisDto.setLogoUrl(analysis.getLogoUrl());
        analysisDto.setPrizeUrl(analysis.getPrizeUrl());
        analysisDto.setAuthUser(analysis.getAuthUser());
        analysisDto.setRule(analysis.getRule());

        return analysisDto;
    }

    @Override
    public List<AnalysisRoundDto> getRoundListByAnalysisId(Integer analysisId) {
        List<AnalysisRound> roundList = analysisRoundMapper.selectRoundListById(analysisId);

        if (null == roundList){
            return null;
        }

        List<AnalysisRoundDto> roundDtoList = new ArrayList<>();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        for (AnalysisRound round : roundList) {
            AnalysisRoundDto roundDto = new AnalysisRoundDto();

            roundDto.setRoundId(round.getId());
            roundDto.setAnalysisId(round.getAnalysisId());
            roundDto.setName(round.getRoundName());
            roundDto.setIntro(round.getIntro());
            roundDto.setStartTime(sdf.format(round.getStartTime()));
            roundDto.setEndTime(sdf.format(round.getEndTime()));
            roundDto.setSeqNo(round.getSeqNo());
            roundDto.setPublished(round.getIsPublished() ? 1 : 0);

            roundDtoList.add(roundDto);
        }

        return roundDtoList;
    }

    @Override
    public RoundExamTitleDto getRoundById(Integer roundId) {
        AnalysisRound round = analysisRoundMapper.selectByPrimaryKey(roundId);

        if (null == round) {
            return null;
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        RoundExamTitleDto roundExamTitleDto = new RoundExamTitleDto();

        roundExamTitleDto.setAnalysisId(round.getAnalysisId());
        roundExamTitleDto.setRoundId(round.getId());
        roundExamTitleDto.setStartTime(sdf.format(round.getStartTime()));
        roundExamTitleDto.setEndTime(sdf.format(round.getEndTime()));
        roundExamTitleDto.setTotalCount(analysisRoundExamTitleMapper.selectExamTitleCountByRoundId(roundId));
        roundExamTitleDto.setPublished(round.getIsPublished() ? 1: 0);

        return roundExamTitleDto;
    }

    @Override
    public List<ExamTitleDto> getExamTitleListByRoundId(Integer roundId) {
        List<AnalysisRoundExamTitle> roundExamTitleList = analysisRoundExamTitleMapper.selectExamTitleListByRoundId(roundId);

        if (null == roundExamTitleList) {
            return null;
        }

        List<ExamTitleDto> ExamTitleDtoList = new ArrayList<>();
        for (AnalysisRoundExamTitle roundExamTitle : roundExamTitleList) {
            ExamTitleDto examTitleDto = getExamTitleDtoById(roundExamTitle.getId());
            ExamTitleDtoList.add(examTitleDto);
        }

        return ExamTitleDtoList;
    }

    @Override
    public AnalysisRoundExamTitle getAnanlysisRoundExamTitleById(Integer analysisRoundExamTitleId) {

        return analysisRoundExamTitleMapper.selectByPrimaryKey(analysisRoundExamTitleId);
    }

    @Override
    public ExamTitleDto getExamTitleDtoById(Integer analysisRoundExamTitleId) {
        AnalysisRoundExamTitle analysisRoundExamTitle = analysisRoundExamTitleMapper.selectByPrimaryKey(analysisRoundExamTitleId);

        ExamTitleDto examTitleDto = new ExamTitleDto();

        examTitleDto.setExamTitleId(analysisRoundExamTitle.getId());
        examTitleDto.setSeqNo(analysisRoundExamTitle.getSeqNo());
        examTitleDto.setType(analysisRoundExamTitle.getExamTitleType());
        examTitleDto.setQuestion(analysisRoundExamTitle.getQuestion());

        /**
         * 设置选项
         */
        List<String> configList = new ArrayList<>();
        String[] configArr = analysisRoundExamTitle.getConfig().split("\\$\\{answer\\}");
        for (int idx = 0; idx < configArr.length; ++ idx) {
            configList.add(OPTIONS[idx] + "." + configArr[idx]);
        }

        examTitleDto.setConfig(configList);

        /**
         * 设置答案
         */
        examTitleDto.setAnswer(analysisRoundExamTitle.getAnswer());
        String[] answerArr = analysisRoundExamTitle.getAnswer().split(",");
        List<Integer> idxList = new ArrayList<>();

        for (String sa : answerArr) {
            for (int i = 0;  i < 8; ++ i) {
                if (sa.equals(OPTIONS[i])) {
                    idxList.add(i);
                }
            }
        }
        examTitleDto.setAnswerIdx(StringUtils.join(idxList, ","));

        examTitleDto.setErrorRate(analysisRoundExamTitle.getErrorRate());
        examTitleDto.setStarCount(starRecordMapper.selectStarCountByRoundExamTitleId(analysisRoundExamTitle.getId()));
        examTitleDto.setReplyCount(replyMapper.selectReplyCountByAnalysisRoundExamTitleId(analysisRoundExamTitle.getId()));

        return examTitleDto;
    }

    @Override
    public List<ReplyDto> getReplyListDtoByExamTitleId(Integer analysisRoundExamTitleId, Integer userId) {
        List<Reply> replyList = replyMapper.selectReplyListByAnalysisRoundExamTitleId(analysisRoundExamTitleId);

        List<ReplyDto> replyDtoList = new ArrayList<>();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        for (Reply reply : replyList) {
            replyDtoList.add(getReplyDtoByReply(reply, userId));
        }

        return replyDtoList;
    }

    @Override
    public Reply getReplyById(Integer replyId) {
        return replyMapper.selectByPrimaryKey(replyId);
    }

    @Override
    public ReplyDto getReplyDtoByReply(Reply reply, Integer userId) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        ReplyDto replyDto = new ReplyDto();
        Doctor doctor = doctorMapper.selectByPrimaryKey(reply.getUserId());

        replyDto.setReplyId(reply.getId());
        replyDto.setSeqNo(reply.getSeqNo());
        replyDto.setUserId(reply.getUserId());
        replyDto.setUserAvatarImageUrl(doctor.getAvatarImageUrl());
        replyDto.setUserName(doctor.getName());
        replyDto.setPostTitle(doctor.getTitle());
        replyDto.setHospital(doctor.getHospital());
        replyDto.setContent(reply.getContent());
        replyDto.setReplayTime(sdf.format(reply.getCreatedTime()));
        replyDto.setStarCount(starRecordMapper.selectStarCountByReplyId(reply.getId()));

        StarRecord starRecord = new StarRecord();
        starRecord.setReplyId(reply.getId());
        starRecord.setCreatedId(userId);

        replyDto.setStatus(starRecordMapper.selectStarCountByRecord(starRecord) > 0 ? 1: 0);

        return replyDto;
    }

    @Override
    public Integer insertStarRecord(Integer analysisId, Integer roundId, Integer examTitleId, Integer replyId, Integer userId) {
        StarRecord starRecord = new StarRecord();

        starRecord.setAnalysisId(analysisId);
        starRecord.setRoundId(roundId);
        starRecord.setAnalysisRoundExamTitleId(examTitleId);
        starRecord.setReplyId(replyId);
        starRecord.setCreatedId(userId);

        /**
         * 1. 判断是否为同一条回复点过赞
         */
        if (starRecordMapper.selectStarCountByRecord(starRecord) > 0) {
            return -1;
        }

        /**
         * 2. 插入点赞记录
         */
        starRecord.setCreatedTime(new Date());
        starRecord.setModifiedId(userId);
        starRecord.setModifiedTime(new Date());
        starRecord.setIsDeleted(false);

        starRecordMapper.insert(starRecord);

        /**
         * 3. 修改点赞数
         */
        Reply reply = replyMapper.selectByPrimaryKey(replyId);
        reply.setStarCount(reply.getStarCount() + 1);
        replyMapper.updateByPrimaryKey(reply);

        return 0;
    }

    @Override
    public Integer removeStarRecord(Integer replyId, Integer userId) {

        /**
         * 1. 判断是否存在点赞记录
         */
        StarRecord starRecord = new StarRecord();
        starRecord.setReplyId(replyId);
        starRecord.setCreatedId(userId);
        if (starRecordMapper.selectStarCountByRecord(starRecord) == 0) {
            return -1;
        }

        /**
         * 2. 删除点赞记录
         */
        StarRecord record = starRecordMapper.selectBySelective(starRecord);
        record.setModifiedId(userId);
        record.setModifiedTime(new Date());
        record.setIsDeleted(true);

        starRecordMapper.updateByPrimaryKeySelective(record);

        /**
         * 3. 修改点赞数
         */
        Reply reply = replyMapper.selectByPrimaryKey(replyId);
        reply.setStarCount(reply.getStarCount() - 1);
        replyMapper.updateByPrimaryKey(reply);

        return 0;
    }
}
