package com.pica.cloud.online.exam.analysis.server.controller;

import com.alibaba.fastjson.JSONObject;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.utils.annotation.LoginPermission;
import com.pica.cloud.foundation.utils.controller.BaseController;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.online.exam.analysis.common.dto.*;
import com.pica.cloud.online.exam.analysis.common.util.ReturnUtil;
import com.pica.cloud.online.exam.analysis.server.configuration.PropertiesConfiguration;
import com.pica.cloud.online.exam.analysis.server.service.CHCAnalysisService;
import com.pica.cloud.online.exam.analysis.server.service.DoctorService;
import com.pica.cloud.foundation.redis.ICacheClient;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.bind.annotation.*;
import java.util.Optional;

/**
 * @author wuminghao
 * @date 2018/8/16 18:10
 */
@RestController
@RequestMapping("/")
public class AnalysisController extends BaseController {

    @Qualifier("cacheMigrateClient")
    @Autowired
    private ICacheClient redisClient;

    @Autowired
    private CHCAnalysisService analysisService;
    @Autowired
    private PropertiesConfiguration propertiesConfiguration;

    static final Integer CHC_ANALYSIS_ID = 1;
    static final Integer PSA_ANALYSIS_ID = 2;

    @RequestMapping(value = "/config", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public String config() {
        return propertiesConfiguration.toString();
    }

    @RequestMapping(value = {"/activityDetail", "/activityDetail/{id}"}, method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<CHCAnalysisDto> getActivityDetail(@PathVariable(required = false) Integer id) {
        PicaResponse.Builder<CHCAnalysisDto> builder = new PicaResponse.Builder<>();
        PicaUser user = null;
        try {
            user = fetchPicaUser();
        } catch (Exception e) {
        }
        builder.setData(analysisService.getCHCActivityDetail(id == null ? CHC_ANALYSIS_ID : id, user));
        return builder.build();
    }

    @RequestMapping(value = "/psaActivityDetail", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<CHCAnalysisDto> getPSAActivityDetail(@RequestParam(required = false) Integer id) {
        PicaResponse.Builder<CHCAnalysisDto> builder = new PicaResponse.Builder<>();
        builder.setData(analysisService.getPSAActivityDetail(id == null ? PSA_ANALYSIS_ID : id));
        return builder.build();
    }

    @RequestMapping(value = "/roundDetail/{roundId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<RoundExamTitleDto> getRoundDetail(@PathVariable("roundId") Integer roundId) {
        PicaResponse.Builder<RoundExamTitleDto> builder = new PicaResponse.Builder<>();
        builder.setData(analysisService.getRoundDetail(roundId));
        return builder.build();
    }

    @RequestMapping(value = "/replyList/{examTitleId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDto> getReplyList(@PathVariable("examTitleId") Integer examTitleId) {
        PicaResponse.Builder<ExamTitleReplyDto> builder = new PicaResponse.Builder<>();
        PicaUser user = null;
        try {
            user = fetchPicaUser();
        } catch (Exception e) {
        }
        builder.setData(analysisService.getReplyListByExamTitleId(examTitleId, user));
        return builder.build();
    }

    @RequestMapping(value = "/replyList/v2/{examTitleId}/page/{pageNo}/size/{pageSize}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDto> getReplyList(@PathVariable("examTitleId") Integer examTitleId,
                                                        @PathVariable("pageNo") Integer pageNo,
                                                        @PathVariable("pageSize") Integer pageSize) {
        PicaResponse.Builder<ExamTitleReplyDto> builder = new PicaResponse.Builder<>();
        PicaUser user = null;
        try {
            user = fetchPicaUser();
        } catch (Exception e) {
        }
        builder.setData(analysisService.getReplyListV2ByExamTitleId(examTitleId, pageNo, pageSize, user));
        return builder.build();
    }

    @RequestMapping(value = "/replyDetail/{replyId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDetailDto> getReplyDetail(@PathVariable("replyId") Integer replyId) {
        PicaResponse.Builder<ExamTitleReplyDetailDto> builder = new PicaResponse.Builder<>();
        PicaUser user = null;
        try {
            user = fetchPicaUser();
        } catch (Exception e) {
        }
        builder.setData(analysisService.getReplyDetailByReplyId(replyId, user));
        return builder.build();
    }

    @RequestMapping(value = "/replyDetailByExamTitleId/{examTitleId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDetailDto> getReplyDetailByExamTitleId(@PathVariable("examTitleId") Integer examTitleId) {
        PicaResponse.Builder<ExamTitleReplyDetailDto> builder = new PicaResponse.Builder<>();
        PicaUser user = null;
        try {
            user = fetchPicaUser();
        } catch (Exception e) {
        }
        builder.setData(analysisService.getReplyDetailByExamTitleId(examTitleId, user));
        return builder.build();
    }

    @LoginPermission
    @RequestMapping(value = "/starReply/{replyId}/analysis/{analysisId}/round/{roundId}/examTitle/{examTitleId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse starReply(@PathVariable("analysisId") Integer analysisId,
                                  @PathVariable("roundId") Integer roundId,
                                  @PathVariable("examTitleId") Integer examTitleId,
                                  @PathVariable("replyId") Integer replyId) {
        return analysisService.starReply(fetchPicaUser(), analysisId, roundId, examTitleId, replyId);
    }

    @LoginPermission
    @RequestMapping(value = "/cancelStar/{replyId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse cancelStar( @PathVariable("replyId") Integer replyId,
                                    @RequestHeader(required = false) String token) {
        if (0 != analysisService.removeStarRecord(replyId, fetchPicaUser())) {
            return ReturnUtil.getPicaResponse(PicaResultCode.RESULE_DATA_NONE);
        }
        return ReturnUtil.getPicaResponse(PicaResultCode.SUCCESS);
    }

    @RequestMapping(value = "/nextExamTitle/{examTitleId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDto> nextReplyList(@PathVariable("examTitleId") Integer examTitleId) {
        PicaResponse.Builder<ExamTitleReplyDto> builder = new PicaResponse.Builder<>();
        PicaUser user = null;
        try {
            user = fetchPicaUser();
        } catch (Exception e) {
        }
        builder.setData(analysisService.getNextReplyListByExamTitleId(examTitleId, user));
        return builder.build();
    }

    @RequestMapping(value = "/nextReply/{replyId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<ExamTitleReplyDetailDto> nextReplyDetail(@PathVariable("replyId") Integer replyId) {
        PicaResponse.Builder<ExamTitleReplyDetailDto> builder = new PicaResponse.Builder<>();
        PicaUser user = null;
        try {
            user = fetchPicaUser();
        } catch (Exception e) {
        }
        builder.setData(analysisService.getNextReplyDetailByReplyId(replyId, user));
        return builder.build();
    }

    @LoginPermission
    @RequestMapping(value = "/reply/{examTitleId}", method = RequestMethod.POST, produces = "application/json;charset=UTF-8")
    public PicaResponse reply( @PathVariable("examTitleId") Integer examTitleId,
                               @RequestBody String content,
                               @RequestHeader String sysCode) {
        JSONObject jsonObject = JSONObject.parseObject(content);
        content = jsonObject.get("content").toString();
        return analysisService.reply(fetchPicaUser(), sysCode, examTitleId, content,false);
    }

    @ApiOperation(value = "获取某一轮的题目V2", response = PicaResponse.class)
    @RequestMapping(value = "/roundDetail/v2/{roundId}", method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<RoundExamTitleV2Dto> getRoundDetailV2(@PathVariable(value = "roundId") Integer roundId) {
        PicaResponse.Builder<RoundExamTitleV2Dto> builder = new PicaResponse.Builder<>();
        builder.setData(analysisService.getRoundDetailV2(roundId));
        return builder.build();
    }

    @LoginPermission
    @ApiOperation(value = "我的赞", response = PicaResponse.class)
    @RequestMapping(value = {"/myStar/{roundId}","/myStar/"}, method = RequestMethod.GET, produces = "application/json;charset=UTF-8")
    public PicaResponse<MyStarDto> myStar( @PathVariable(value = "roundId", required = false) Integer roundId) {

        PicaResponse.Builder<MyStarDto> builder = new PicaResponse.Builder<>();
        builder.setData(analysisService.myStar(fetchPicaUser(), Optional.ofNullable(roundId)));
        return builder.build();
    }

}
