package com.pica.cloud.online.exam.analysis.server.service.impl;

import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.utils.entity.PicaUser;
import com.pica.cloud.online.exam.analysis.common.constants.CommonConstants;
import com.pica.cloud.online.exam.analysis.common.dto.*;
import com.pica.cloud.online.exam.analysis.common.util.ReturnUtil;
import com.pica.cloud.online.exam.analysis.server.configuration.PropertiesConfiguration;
import com.pica.cloud.online.exam.analysis.server.entity.*;
import com.pica.cloud.online.exam.analysis.server.mapper.*;
import com.pica.cloud.online.exam.analysis.server.service.AntiSpamService;
import com.pica.cloud.online.exam.analysis.server.service.CHCAnalysisService;
import com.pica.cloud.online.exam.analysis.server.service.CHCRankingListService;
import com.pica.cloud.online.exam.analysis.server.service.DoctorService;
import com.pica.cloud.online.exam.analysis.server.utils.DateUtils;
import net.bytebuddy.asm.Advice;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author wuminghao
 * @date 2018/8/17 9:45
 */
@Service
public class CHCAnalysisServiceImpl implements CHCAnalysisService {
    static final String[] OPTIONS = {"A", "B", "C", "D", "E", "F", "G", "H"};
    private static Logger logger = LoggerFactory.getLogger(CHCAnalysisServiceImpl.class);

    @Autowired
    CHCAnalysisMapper analysisMapper;

    @Autowired
    AnalysisRoundMapper analysisRoundMapper;

    @Autowired
    AnalysisRoundExamTitleMapper analysisRoundExamTitleMapper;

    @Autowired
    StarRecordMapper starRecordMapper;

    @Autowired
    ReplyMapper replyMapper;

    @Autowired
    DoctorMapper doctorMapper;

    @Autowired
    CHCRankingListMapper rankingListMapper;

    @Autowired
    private PropertiesConfiguration properties;

    @Autowired
    private CHCRankingListService chcRankingListService;

    @Autowired
    private DoctorService doctorService;

    @Autowired
    private AntiSpamService antiSpamService;

    static final byte ANALYSIS_TYPE_CHC = 1;
    static final byte ANALYSIS_TYPE_PSA = 2;

    static final int CHC_ANALYSIS_ID = 1;
    static final int PSA_ANALYSIS_ID = 2;

    /**
     * chc 活动详情
     * @param analysisId
     * @param user
     * @return
     */
    @Override
    public CHCAnalysisDto getCHCActivityDetail(Integer analysisId, PicaUser user) {
        CHCAnalysisDto analysisDto = this.getCHCAnalysisDtoById(analysisId);
        if (chcRankingListService.isRankingInTop200(user != null ? user.getId() : 0)) {
            analysisDto.setIsJoinIn(1);
        } else {
            analysisDto.setIsJoinIn(0);
        }
        List<AnalysisRoundDto> roundList = this.getRoundListByAnalysisId(analysisId);

        if (analysisId == CHC_ANALYSIS_ID) {
            /**
             * chc活动固定返回两轮, 将第一轮设置为所有
             */
            if (roundList.size() > 0) {
                roundList.get(0).setRoundId(-1);
            }
            /**
             * 如果是最后一起, 补上最后一期 和前端约定为-2
             */
            if (roundList.size() == 1) {
                AnalysisRoundDto analysisRoundDto = new AnalysisRoundDto();
                analysisRoundDto.setRoundId(-2);
                roundList.add(analysisRoundDto);
            }
        }
        analysisDto.setRoundList(roundList);

        return analysisDto;
    }

    @Override
    public CHCAnalysisDto getPSAActivityDetail(Integer analysisId) {
        CHCAnalysisDto analysisDto = getPSAAnalysisDtoById(analysisId);
        List<AnalysisRoundDto> roundList = getRoundListByAnalysisId(analysisId);
        analysisDto.setRoundList(roundList);
        return analysisDto;
    }

    @Override
    public RoundExamTitleDto getRoundDetail(Integer roundId) {
        RoundExamTitleDto roundExamTitleDto = getRoundExamTitleDtoById(roundId);
        List<ExamTitleDto> examTitleDtoList = getExamTitleListByRoundId(roundId);
        roundExamTitleDto.setExamTitleList(examTitleDtoList);
        return roundExamTitleDto;
    }

    @Override
    public RoundExamTitleV2Dto getRoundDetailV2(Integer roundId) {
        RoundExamTitleV2Dto roundExamTitleV2Dto = new RoundExamTitleV2Dto();
        /**
         * 有rounfid传入时设置roundList为空
         */
        if (roundId > 0) {
            roundExamTitleV2Dto.setRoundList(new ArrayList<>());
        } else {
            List<AnalysisRoundDto> roundDtoList = getCHCHistoryRound(CHC_ANALYSIS_ID);
            /**
             * 获取list中最后一个的roundId
             */
            if (roundDtoList.size() > 0) {
                roundId = roundDtoList.get(roundDtoList.size() - 1).getRoundId();
            }
            roundExamTitleV2Dto.setRoundList(roundDtoList);
        }
        RoundExamTitleDto roundExamTitleDto = getRoundExamTitleDtoById(roundId);

        try {
            /**
             * 转换格式
             */
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            SimpleDateFormat outFormat = new SimpleDateFormat("yyyy-M-d");
            roundExamTitleV2Dto.setAnalysisId(roundExamTitleDto.getAnalysisId());
            roundExamTitleV2Dto.setRoundId(roundExamTitleDto.getRoundId());
            roundExamTitleV2Dto.setStartTime(outFormat.format(sdf.parse(roundExamTitleDto.getStartTime())));
            roundExamTitleV2Dto.setEndTime(outFormat.format(sdf.parse(roundExamTitleDto.getEndTime())));
            roundExamTitleV2Dto.setTotalCount(roundExamTitleDto.getTotalCount());
            roundExamTitleV2Dto.setPublished(roundExamTitleDto.getPublished());

            if ((new Date()).getTime() > sdf.parse(roundExamTitleDto.getEndTime()).getTime()) {
                roundExamTitleV2Dto.setRemainingTime("");
                roundExamTitleV2Dto.setIsFinished(1);
                roundExamTitleV2Dto.setElectedStarCount(getElectedStarCountByRoundId(roundId));

                Integer doctorId = getElectedDoctorIdByRoundId(roundId);
                roundExamTitleV2Dto.setElectedDoctor(doctorId != null ? doctorService.getDoctorDtoById(doctorId) : new DoctorDto());
            } else {
                roundExamTitleV2Dto.setRemainingTime(DateUtils.remainingTime(sdf.parse(roundExamTitleDto.getEndTime())));
                roundExamTitleV2Dto.setIsFinished(0);
                roundExamTitleV2Dto.setElectedDoctor(new DoctorDto());
            }
        } catch (Exception e) {
            logger.error("getRoundDetailV2: ", e);
        }

        List<ExamTitleDto> examTitleDtoList = getExamTitleListByRoundId(roundId);
        roundExamTitleV2Dto.setExamTitleList(examTitleDtoList);

        return roundExamTitleV2Dto;
    }

    @Override
    public ExamTitleReplyDto getReplyListByExamTitleId(Integer examTitleId, PicaUser user) {
        AnalysisRoundExamTitle analysisRoundExamTitle = analysisRoundExamTitleMapper.selectByPrimaryKey(examTitleId);
        CHCAnalysis analysis = analysisMapper.selectByPrimaryKey(analysisRoundExamTitle.getAnalysisId());

        ExamTitleReplyDto examTitleReplyDto = new ExamTitleReplyDto();
        ExamTitleDto examTitleDto = getExamTitleDtoById(examTitleId);
        List<ReplyDto> replyDtoList = getReplyListDtoByExamTitleId(examTitleId, user !=  null ? user.getId() : 0);

        /**
         * 如果是PSA 删除回复的时间显示
         */
        if (analysis.getType() == ANALYSIS_TYPE_PSA ) {
            replyDtoList.forEach(replyDto -> replyDto.setReplayTime(""));
        }

        examTitleReplyDto.setAnalysisId(analysisRoundExamTitle.getAnalysisId());
        examTitleReplyDto.setRoundId(analysisRoundExamTitle.getRoundId());
        examTitleReplyDto.setExamTitle(examTitleDto);
        examTitleReplyDto.setReplyList(replyDtoList);
        examTitleReplyDto.setType(analysis.getType().intValue());

        AnalysisRound analysisRound = analysisRoundMapper.selectByPrimaryKey(analysisRoundExamTitle.getRoundId());
        examTitleReplyDto.setPublished(analysisRound.getIsPublished() ? 1 : 0);
        /**
         * 设置回复状态
         */
        examTitleReplyDto.setReplyStatus(0);
        if ((new Date()).getTime() < analysisRound.getEndTime().getTime()) {
            if (user != null && user.getId() != 0) {
                examTitleReplyDto.setReplyStatus(getReplyStatus(analysis, examTitleId, user.getId()));
            }
        }
        return examTitleReplyDto;
    }


    @Override
    public ExamTitleReplyDto getNextReplyListByExamTitleId(Integer examTitleId, PicaUser user) {
        AnalysisRoundExamTitle analysisRoundExamTitle = analysisRoundExamTitleMapper.selectByPrimaryKey(examTitleId);
        ExamTitleDto examTitleDto = getNextExamTitleDtoById(analysisRoundExamTitle.getRoundId(), examTitleId);

        return getReplyListByExamTitleId(examTitleDto.getExamTitleId(),  user);
    }

    @Override
    public ExamTitleReplyDetailDto getReplyDetailByReplyId(Integer replyId, PicaUser user) {

        ExamTitleReplyDetailDto examTitleReplyDetailDto = new ExamTitleReplyDetailDto();

        Reply reply = replyMapper.selectByPrimaryKey(replyId);
        AnalysisRoundExamTitle analysisRoundExamTitle = analysisRoundExamTitleMapper.selectByPrimaryKey(reply.getAnalysisRoundExamTitleId());

        ExamTitleDto examTitleDto = getExamTitleDtoById(reply.getAnalysisRoundExamTitleId());
        ReplyDto replyDto = getReplyDtoByReply(reply, user != null ? user.getId() : 0);

        CHCAnalysis analysis = analysisMapper.selectByPrimaryKey(analysisRoundExamTitle.getAnalysisId());

        examTitleReplyDetailDto.setAnalysisId(analysisRoundExamTitle.getAnalysisId());
        examTitleReplyDetailDto.setRoundId(analysisRoundExamTitle.getRoundId());
        examTitleReplyDetailDto.setExamTitle(examTitleDto);
        examTitleReplyDetailDto.setReply(replyDto);
        examTitleReplyDetailDto.setType(analysis.getType().intValue());

        AnalysisRound analysisRound = analysisRoundMapper.selectByPrimaryKey(analysisRoundExamTitle.getRoundId());
        examTitleReplyDetailDto.setPublished(analysisRound.getIsPublished() ? 1 : 0);
        /**
         * 设置回复状态
         */
        examTitleReplyDetailDto.setReplyStatus(0);
        if ((new Date()).getTime() < analysisRound.getEndTime().getTime()) {
            if (user != null && user.getId() != 0) {
                examTitleReplyDetailDto.setReplyStatus(getReplyStatus(analysis, reply.getAnalysisRoundExamTitleId(), user.getId()));
            }
        }
        return examTitleReplyDetailDto;
    }

    @Override
    public ExamTitleReplyDetailDto getNextReplyDetailByReplyId(Integer replyId, PicaUser user) {
        Reply reply = getNextReplyById(replyId);
        return getReplyDetailByReplyId(reply.getId(), user);
    }

    @Override
    public ExamTitleReplyDetailDto getReplyDetailByExamTitleId(Integer examTitleId, PicaUser user) {
        ExamTitleReplyDetailDto examTitleReplyDetailDto = new ExamTitleReplyDetailDto();

        Reply reply =  getReplyByExamTitleId(examTitleId, user != null ? user.getId() : 0);
        AnalysisRoundExamTitle analysisRoundExamTitle = analysisRoundExamTitleMapper.selectByPrimaryKey(examTitleId);
        ExamTitleDto examTitleDto = getExamTitleDtoById(examTitleId);

        ReplyDto replyDto = new ReplyDto();
        if (null != reply) {
            replyDto = getReplyDtoByReply(reply, user != null ? user.getId() : 0);
        } else {
            replyDto.setContent("");
        }
        examTitleReplyDetailDto.setAnalysisId(analysisRoundExamTitle.getAnalysisId());
        examTitleReplyDetailDto.setRoundId(analysisRoundExamTitle.getRoundId());
        examTitleReplyDetailDto.setExamTitle(examTitleDto);
        examTitleReplyDetailDto.setReply(replyDto);

        AnalysisRound analysisRound = analysisRoundMapper.selectByPrimaryKey(analysisRoundExamTitle.getRoundId());
        examTitleReplyDetailDto.setPublished(analysisRound.getIsPublished() ? 1 : 0);

        CHCAnalysis analysis = analysisMapper.selectByPrimaryKey(analysisRoundExamTitle.getAnalysisId());
        examTitleReplyDetailDto.setType(analysis.getType().intValue());

        /**
         * 设置回复状态
         */
        examTitleReplyDetailDto.setReplyStatus(0);
        if ((new Date()).getTime() < analysisRound.getEndTime().getTime()) {
            if (user != null && user.getId() != 0) {
                examTitleReplyDetailDto.setReplyStatus(getReplyStatus(analysis, reply.getAnalysisRoundExamTitleId(), user.getId()));
            }
        }
        return examTitleReplyDetailDto;
    }

    @Override
    public PicaResponse starReply(PicaUser user, Integer analysisId, Integer roundId, Integer examTitleId, Integer replyId) {
        Doctor doctor = doctorService.getDoctorById(user.getId());
        /**
         * 未认证用户直接返回
         */
        if (!(doctor.getStatus().equals(CommonConstants.DOCTOR_STATUS_3) ||
                doctor.getStatus().equals(CommonConstants.DOCTOR_STATUS_6) ||
                doctor.getStatus().equals(CommonConstants.DOCTOR_STATUS_7) )) {
            return ReturnUtil.getPicaResponse(PicaResultCode.INTERFACE_FORBID_VISIT);
        }

        AnalysisRound analysisRound = analysisRoundMapper.selectByPrimaryKey(roundId);
        if ((new Date()).getTime() > analysisRound.getEndTime().getTime()) {
            return ReturnUtil.getPicaResponse("500002", "本期活动已结束");
        }

        /**
         * 已经点过赞
         */
        if (0 != insertStarRecord(analysisId, roundId, examTitleId, replyId, user.getId())) {
            return ReturnUtil.getPicaResponse(PicaResultCode.DATA_ALREADY_EXISTED);
        }
        return ReturnUtil.getPicaResponse(PicaResultCode.SUCCESS);
    }


    @Override
    public PicaResponse reply(PicaUser user, String sysCode, Integer examTitleId, String content, Boolean isAdFilterOpen) {
        /**
         * 如果判断检测的数据中有垃圾 直接返回
         */
        PicaResponse response = antiSpamService.processString(user.getId(), sysCode, content, false);
        if (!response.getCode().equals(PicaResultCode.SUCCESS.code())) {
            response.setData("");
            return response;
        }
        insertReply(examTitleId, content, user.getId());

        PicaResponse picaResponse = ReturnUtil.getPicaResponse(PicaResultCode.SUCCESS);
        picaResponse.setMessage("发布成功");
        return picaResponse;
    }

    @Override
    public MyStarDto myStar(PicaUser user, Optional<Integer> roundId) {

        MyStarDto myStarDto = new MyStarDto();
        myStarDto.setDoctor(doctorService.getDoctorDtoById(user.getId()));

        myStarDto.setRoundList(new ArrayList<>());
        if (!roundId.isPresent()) {
            List<AnalysisRoundDto> roundDtoList = getCHCHistoryRound(CHC_ANALYSIS_ID);
            roundId = Optional.of(roundDtoList.get(0).getRoundId());
            myStarDto.setRoundList(roundDtoList);
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy.M.d");
        AnalysisRound analysisRound = analysisRoundMapper.selectByPrimaryKey(roundId.get());
        myStarDto.setStartTime(sdf.format(analysisRound.getStartTime()));
        myStarDto.setEndTime(sdf.format(analysisRound.getEndTime()));
        myStarDto.setRemainingTime(DateUtils.remainingTime(analysisRound.getEndTime()));
        /**
         * 设置结束状态
         */
        if ((new Date()).getTime() > analysisRound.getEndTime().getTime()) {
            myStarDto.setIsFinished(1);
            /**
             * 设置我是否当选
             */
            Integer electedDoctorId = getElectedDoctorIdByRoundId(roundId.get());
            myStarDto.setIsElected(electedDoctorId.intValue() == user.getId().intValue() ? 1 : 0);
        } else {
            myStarDto.setIsFinished(0);
            myStarDto.setIsElected(0);
        }

        /**
         * 设置我的得赞数
         */
        myStarDto.setMyStarCount(getStarCountByRoundIdAndDoctorId(roundId.get(), user.getId()));
        /**
         * 我的解析得赞记录
         */
        myStarDto.setMyExamTitleList(getMyStarRecordByRoundIdAndDoctorId(roundId.get(), user.getId()));
        return myStarDto;
    }

    @Override
    public Integer removeStarRecord(Integer replyId, PicaUser user) {
        /**
         * 1. 判断是否存在点赞记录
         */
        StarRecord starRecord = new StarRecord();
        starRecord.setReplyId(replyId);
        starRecord.setCreatedId(user.getId());
        if (starRecordMapper.selectStarCountByRecord(starRecord) == 0) {
            return -1;
        }

        /**
         * 2. 删除点赞记录
         */
        StarRecord record = starRecordMapper.selectBySelective(starRecord);
        record.setModifiedId(user.getId());
        record.setModifiedTime(new Date());
        record.setIsDeleted(true);

        starRecordMapper.updateByPrimaryKeySelective(record);

        /**
         * 3. 修改点赞数
         */
        Reply reply = replyMapper.selectByPrimaryKey(replyId);
        reply.setStarCount(reply.getStarCount() - 1);
        replyMapper.updateByPrimaryKey(reply);

        return 0;
    }
    //////////////////////////////////////////////////////////////////////////////////////////////////////
    private CHCAnalysisDto getCHCAnalysisDtoById(Integer analysisId) {

        CHCAnalysis analysis = analysisMapper.selectCHCByPrimaryKey(analysisId);
        if (null ==  analysis) {
            return null;
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        CHCAnalysisDto analysisDto = new CHCAnalysisDto();
        analysisDto.setId(analysis.getId());
        analysisDto.setName(analysis.getAnalysisName());
        analysisDto.setMainTitle(analysis.getMainTitle());
        analysisDto.setSubTitle(analysis.getSubTitle());
        analysisDto.setIntro(analysis.getIntro());
        analysisDto.setStartTime(sdf.format(analysis.getStartTime()));
        analysisDto.setEndTime(sdf.format(analysis.getEndTime()));
        analysisDto.setLogoUrl(analysis.getLogoUrl());
        analysisDto.setPrizeUrl(analysis.getPrizeUrl());
        analysisDto.setAuthUser(analysis.getAuthUser());
        analysisDto.setRule(analysis.getRule());

        return analysisDto;
    }

    private CHCAnalysisDto getPSAAnalysisDtoById(Integer analysisId) {
        CHCAnalysis analysis = analysisMapper.selectPSAByPrimaryKey(analysisId);
        if (null ==  analysis) {
            return null;
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        CHCAnalysisDto analysisDto = new CHCAnalysisDto();
        analysisDto.setId(analysis.getId());
        analysisDto.setName(analysis.getAnalysisName());
        analysisDto.setMainTitle(analysis.getMainTitle());
        analysisDto.setSubTitle(analysis.getSubTitle());
        analysisDto.setIntro(analysis.getIntro());
        analysisDto.setStartTime(sdf.format(analysis.getStartTime()));
        analysisDto.setEndTime(sdf.format(analysis.getEndTime()));
        analysisDto.setLogoUrl(analysis.getLogoUrl());
        analysisDto.setPrizeUrl(analysis.getPrizeUrl());
        analysisDto.setAuthUser(analysis.getAuthUser());
        analysisDto.setRule(analysis.getRule());

        return analysisDto;
    }

    private List<AnalysisRoundDto> getRoundListByAnalysisId(Integer analysisId) {
        List<AnalysisRound> roundList = new ArrayList<AnalysisRound>();

        CHCAnalysis analysis = analysisMapper.selectByPrimaryKey(analysisId);
        if (analysis.getType() == 1) {
            roundList = analysisRoundMapper.selectRoundListById(analysisId);
        } else {
            roundList = analysisRoundMapper.selectPSARoundListById(analysisId);
        }
        if (null == roundList){
            return null;
        }

        List<AnalysisRoundDto> roundDtoList = new ArrayList<>();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        for (AnalysisRound round : roundList) {
            AnalysisRoundDto roundDto = new AnalysisRoundDto();

            roundDto.setRoundId(round.getId());
            roundDto.setAnalysisId(round.getAnalysisId());
            roundDto.setName(round.getRoundName());
            roundDto.setIntro(round.getIntro());
            roundDto.setStartTime(sdf.format(round.getStartTime()));
            roundDto.setEndTime(sdf.format(round.getEndTime()));
            roundDto.setSeqNo(round.getSeqNo());

            /**
             * 如果时间过期 设置超时
             */
            if ((new Date()).getTime() > round.getEndTime().getTime()) {
                roundDto.setPublished(2);
            } else {
                roundDto.setPublished(round.getIsPublished() ? 1 : 0);
            }
            roundDtoList.add(roundDto);
        }

        return roundDtoList;
    }

    private RoundExamTitleDto getRoundExamTitleDtoById(Integer roundId) {
        AnalysisRound round = analysisRoundMapper.selectByPrimaryKey(roundId);

        if (null == round) {
            return null;
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        RoundExamTitleDto roundExamTitleDto = new RoundExamTitleDto();

        roundExamTitleDto.setAnalysisId(round.getAnalysisId());
        roundExamTitleDto.setRoundId(round.getId());
        roundExamTitleDto.setStartTime(sdf.format(round.getStartTime()));
        roundExamTitleDto.setEndTime(sdf.format(round.getEndTime()));
        roundExamTitleDto.setTotalCount(analysisRoundExamTitleMapper.selectExamTitleCountByRoundId(roundId));

        /**
         * 如果时间过期 设置超时
         */
        if ((new Date()).getTime() > round.getEndTime().getTime()) {
            roundExamTitleDto.setPublished(2);
        } else {
            roundExamTitleDto.setPublished(round.getIsPublished() ? 1 : 0);
        }
        return roundExamTitleDto;
    }

    private List<ExamTitleDto> getExamTitleListByRoundId(Integer roundId) {
        List<AnalysisRoundExamTitle> roundExamTitleList = analysisRoundExamTitleMapper.selectExamTitleListByRoundId(roundId);

        if (null == roundExamTitleList) {
            return null;
        }

        List<ExamTitleDto> ExamTitleDtoList = new ArrayList<>();
        for (AnalysisRoundExamTitle roundExamTitle : roundExamTitleList) {
            ExamTitleDto examTitleDto = getExamTitleDtoById(roundExamTitle.getId());
            ExamTitleDtoList.add(examTitleDto);
        }

        return ExamTitleDtoList;
    }

    public ExamTitleDto getExamTitleDtoById(Integer analysisRoundExamTitleId) {
        AnalysisRoundExamTitle analysisRoundExamTitle = analysisRoundExamTitleMapper.selectByPrimaryKey(analysisRoundExamTitleId);

        ExamTitleDto examTitleDto = new ExamTitleDto();

        examTitleDto.setExamTitleId(analysisRoundExamTitle.getId());
        examTitleDto.setSeqNo(analysisRoundExamTitle.getSeqNo());
        examTitleDto.setType(analysisRoundExamTitle.getExamTitleType());
        examTitleDto.setQuestion(analysisRoundExamTitle.getSeqNo().toString() + "." + analysisRoundExamTitle.getQuestion());

        /**
         * 设置选项
         */
        if (analysisRoundExamTitle.getConfig() != null) {
            List<String> configList = new ArrayList<>();
            String[] configArr = analysisRoundExamTitle.getConfig().split("\\$\\{answer\\}");
            for (int idx = 0; idx < configArr.length; ++idx) {
                configList.add(OPTIONS[idx] + "." + configArr[idx]);
            }
            examTitleDto.setConfig(configList);
        }


        /**
         * 设置答案
         */
        if (analysisRoundExamTitle.getAnswer() != null) {
            examTitleDto.setAnswer(analysisRoundExamTitle.getAnswer());
            String[] answerArr = analysisRoundExamTitle.getAnswer().split(",");
            List<Integer> idxList = new ArrayList<>();

            for (String sa : answerArr) {
                for (int i = 0; i < 8; ++i) {
                    if (sa.equals(OPTIONS[i])) {
                        idxList.add(i);
                    }
                }
            }
            examTitleDto.setAnswerIdx(StringUtils.join(idxList, ","));
        }
        examTitleDto.setErrorRate(analysisRoundExamTitle.getErrorRate());
        examTitleDto.setStarCount(starRecordMapper.selectStarCountByRoundExamTitleId(analysisRoundExamTitle.getId()));
        examTitleDto.setReplyCount(replyMapper.selectReplyCountByAnalysisRoundExamTitleId(analysisRoundExamTitle.getId()));

        return examTitleDto;
    }

    private List<ReplyDto> getReplyListDtoByExamTitleId(Integer analysisRoundExamTitleId, Integer userId) {
        List<Reply> replyList = getReplyList(analysisRoundExamTitleId);

        List<ReplyDto> replyDtoList = new ArrayList<>();

        for (Reply reply : replyList) {
            try {
                /**
                 * 发现有评论用户不在p_doctor的情况，数据不同步造成的
                 */
                replyDtoList.add(getReplyDtoByReply(reply, userId));
            } catch (Exception e) {
                logger.error("getReplyListDtoByExamTitleId", e);
            }
        }

        return replyDtoList;
    }

    private Reply getNextReplyById(Integer replyId) {
        Reply reply = replyMapper.selectByPrimaryKey(replyId);

        /**
         * 1. 找到该题目下的回复列表,默认是按照点赞排序的
         */
        List<Reply> replyList = getReplyList(reply.getAnalysisRoundExamTitleId());

        /**
         * 2. 找到当前回复的所在位置，返回下一个回复
         */
        int replyCount = replyList.size();
        int idx = 0;
        for (int i = 0; i< replyCount;  ++ i) {
            if (replyList.get(i).getId().intValue() == reply.getId().intValue()) {
                idx = i;
                break;
            }
        }
        idx = ((idx + 1) % replyCount);

        return replyList.get(idx);
    }

    private ReplyDto getReplyDtoByReply(Reply reply, Integer userId) {
        ReplyDto replyDto = new ReplyDto();
        Doctor doctor = doctorMapper.selectByPrimaryKey(reply.getUserId());

        replyDto.setReplyId(reply.getId());
        replyDto.setSeqNo(reply.getSeqNo());
        replyDto.setUserId(reply.getUserId());
        replyDto.setUserAvatarImageUrl(doctor.getAvatarImageUrl() != null
                ? properties.getFileUrl() + doctor.getAvatarImageUrl().trim()
                : properties.getFileUrl() + "/File/doctor_default.png");
        replyDto.setUserName(doctor.getName());
        replyDto.setPostTitle(doctor.getTitle());
        replyDto.setHospital(doctor.getHospital());
        replyDto.setContent(reply.getContent());
        replyDto.setReplayTime(DateUtils.fromNow(reply.getModifiedTime()));
        replyDto.setStarCount(starRecordMapper.selectStarCountByReplyId(reply.getId()));

        AnalysisRoundExamTitle analysisRoundExamTitle = analysisRoundExamTitleMapper.selectByPrimaryKey(reply.getAnalysisRoundExamTitleId());
        AnalysisRound analysisRound = analysisRoundMapper.selectByPrimaryKey(analysisRoundExamTitle.getRoundId());

        if ((new Date()).getTime() > analysisRound.getEndTime().getTime()) {
            replyDto.setStatus(2);
        } else {
            StarRecord starRecord = new StarRecord();
            starRecord.setReplyId(reply.getId());
            starRecord.setCreatedId(userId);

            replyDto.setStatus(starRecordMapper.selectStarCountByRecord(starRecord) > 0 ? 1 : 0);
        }
        return replyDto;
    }

    private Integer insertStarRecord(Integer analysisId, Integer roundId, Integer examTitleId, Integer replyId, Integer userId) {
        StarRecord starRecord = new StarRecord();

        starRecord.setAnalysisId(analysisId);
        starRecord.setRoundId(roundId);
        starRecord.setAnalysisRoundExamTitleId(examTitleId);
        starRecord.setReplyId(replyId);
        starRecord.setCreatedId(userId);

        /**
         * 1. 判断是否为同一条回复点过赞
         */
        if (starRecordMapper.selectStarCountByRecord(starRecord) > 0) {
            return -1;
        }

        /**
         * 2. 插入点赞记录
         */
        starRecord.setCreatedTime(new Date());
        starRecord.setModifiedId(userId);
        starRecord.setModifiedTime(new Date());
        starRecord.setIsDeleted(false);

        starRecordMapper.insert(starRecord);

        /**
         * 3. 修改点赞数
         */
        Reply reply = replyMapper.selectByPrimaryKey(replyId);
        reply.setStarCount(reply.getStarCount() + 1);
        replyMapper.updateByPrimaryKey(reply);

        return 0;
    }

    private ExamTitleDto getNextExamTitleDtoById(Integer roundId, Integer analysisRoundExamTitleId) {

        List<AnalysisRoundExamTitle> roundExamTitleList = analysisRoundExamTitleMapper.selectExamTitleListByRoundId(roundId);
        Integer examTitleCount = roundExamTitleList.size();

        Integer idx = 0;
        for (int i = 0; i < examTitleCount; ++ i) {
            if (roundExamTitleList.get(i).getId().intValue() == analysisRoundExamTitleId.intValue()) {
                idx = i;
                break;
            }
        }
        idx = ((idx + 1) % examTitleCount);

        return this.getExamTitleDtoById(roundExamTitleList.get(idx).getId());
    }

    private Integer insertReply(Integer examTitleId, String content, Integer userId) {
        /**
         * 找到我是否已经回复过,如果回复过就编辑，没有就新建
         */
        Reply reply = new Reply();
        reply.setAnalysisRoundExamTitleId(examTitleId);
        reply.setUserId(userId);

        Integer myReplyCount = replyMapper.selectReplyCountByRecord(reply);

        if (myReplyCount == 0) {
            Integer replyCount = replyMapper.selectReplyCountByAnalysisRoundExamTitleId(examTitleId);

            reply.setContent(content);
            reply.setReplyLevel((byte) 1);
            reply.setSeqNo(replyCount + 1);
            reply.setStarCount(0);
            reply.setCreatedId(userId);
            reply.setCreatedTime(new Date());
            reply.setModifiedId(userId);
            reply.setModifiedTime(new Date());
            reply.setIsDeleted(false);
            return replyMapper.insert(reply);
        } else {
            Reply record = replyMapper.selectReplyByRecord(reply);

            record.setContent(content);
            record.setModifiedTime(new Date());

            return replyMapper.updateByPrimaryKey(record);
        }
    }

    private Integer getReplyStatus(Integer examTitleId, Integer userId) {
        /**
         * 用户不存在
         */
        if (userId == 0) {
            return 0;
        }

        Reply reply = new Reply();
        reply.setAnalysisRoundExamTitleId(examTitleId);
        reply.setUserId(userId);

        Integer myReplyCount = replyMapper.selectReplyCountByRecord(reply);


        if (myReplyCount == 0) {
            /**
             * 未回复过
             */
            return 1;
        }

        /**
         * 已回复过
         */
        return 2;
    }

    private Reply getReplyByExamTitleId(Integer examTitleId, Integer userId) {
        Reply reply = new Reply();
        reply.setAnalysisRoundExamTitleId(examTitleId);
        reply.setUserId(userId);
        return replyMapper.selectReplyByRecord(reply);
    }

    private List<Reply> getReplyList(Integer examTitleId) {
        Integer replyCount = replyMapper.selectReplyCountByAnalysisRoundExamTitleId(examTitleId);

        /**
         * case 1. 按照时间排序，如果个数小于5，直接返回
         */
        List<Reply> replyList = replyMapper.selectReplyListOrderByTime(examTitleId);
        if (replyCount < 5) {
            return replyList;
        }

        /**
         * case 2.大于等于5的时候，根据排序 5最热+ 时间排序
         */
        List<Reply> replyListFive = replyMapper.selectFiveHottestReplyList(examTitleId);

        List<Reply> replyListOther = new ArrayList<>();
        replyListOther.addAll(replyListFive);

        for (Reply reply : replyList) {
            if (!replyListFive.contains(reply))
                replyListOther.add(reply);
        }

        return replyListOther;
    }

    private List<AnalysisRoundDto> getCHCHistoryRound(Integer analysisId) {
        List<AnalysisRound> roundList = new ArrayList<>();

        /**
         * TODO: 需要前端传入id
         */
        roundList = analysisRoundMapper.selectHistoryRoundListById(analysisId);

        List<AnalysisRoundDto> roundDtoList = new ArrayList<>();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        for (AnalysisRound round : roundList) {
            AnalysisRoundDto roundDto = new AnalysisRoundDto();

            roundDto.setRoundId(round.getId());
            roundDto.setAnalysisId(round.getAnalysisId());
            roundDto.setName(round.getRoundName());
            roundDto.setIntro(round.getIntro());
            roundDto.setStartTime(sdf.format(round.getStartTime()));
            roundDto.setEndTime(sdf.format(round.getEndTime()));
            roundDto.setSeqNo(round.getSeqNo());

            /**
             * 如果时间过期 设置超时
             */
            if ((new Date()).getTime() > round.getEndTime().getTime()) {
                roundDto.setPublished(2);
            } else {
                roundDto.setPublished(round.getIsPublished() ? 1 : 0);
            }

            roundDtoList.add(roundDto);
        }

        return roundDtoList;
    }

    private Integer getElectedStarCountByRoundId(Integer roundId) {
        List<AnalysisRoundExamTitle> examTitleList =  analysisRoundExamTitleMapper.selectExamTitleListByRoundId(roundId);
        List<Integer> examTitleIdList = new ArrayList<>();
        for (AnalysisRoundExamTitle examTitle : examTitleList) {
            examTitleIdList.add(examTitle.getId());
        }
        return replyMapper.selectElectedStarCountByExamTitleIdList(examTitleIdList);
    }

    private Integer getElectedDoctorIdByRoundId(Integer roundId) {
        List<AnalysisRoundExamTitle> examTitleList =  analysisRoundExamTitleMapper.selectExamTitleListByRoundId(roundId);
        List<Integer> examTitleIdList = new ArrayList<>();
        for (AnalysisRoundExamTitle examTitle : examTitleList) {
            examTitleIdList.add(examTitle.getId());
        }
        return replyMapper.selectElectedDoctorIdByExamTitleIdList(examTitleIdList);
    }

    private Integer getStarCountByRoundIdAndDoctorId(Integer roundId, Integer doctorId) {
        Map param = new HashMap<>();
        List<AnalysisRoundExamTitle> examTitleList =  analysisRoundExamTitleMapper.selectExamTitleListByRoundId(roundId);
        List<Integer> examTitleIdList = new ArrayList<>();
        for (AnalysisRoundExamTitle examTitle : examTitleList) {
            examTitleIdList.add(examTitle.getId());
        }
        param.put("examTitleList", examTitleIdList);
        param.put("doctorId", doctorId);
        return replyMapper.selectStarCountByParam(param);
    }

    private List<UserExamTitleDto> getMyStarRecordByRoundIdAndDoctorId(Integer roundId, Integer doctorId) {
        Map param = new HashMap<>();
        List<AnalysisRoundExamTitle> examTitleList =  analysisRoundExamTitleMapper.selectExamTitleListByRoundId(roundId);
        List<Integer> examTitleIdList = new ArrayList<>();
        for (AnalysisRoundExamTitle examTitle : examTitleList) {
            examTitleIdList.add(examTitle.getId());
        }
        param.put("examTitleList", examTitleIdList);
        param.put("doctorId", doctorId);

        List<Reply> replyList = replyMapper.selectStarRecordByParam(param);
        List<UserExamTitleDto> userExamTitleDtos = new ArrayList<>();

        for (int idx = 0; idx < examTitleIdList.size(); ++ idx) {
            UserExamTitleDto userExamTitleDto = new UserExamTitleDto();
            userExamTitleDto.setExamTitleId(examTitleIdList.get(idx));
            userExamTitleDto.setSeqNo(idx + 1);

            boolean isReplied = false;
            Integer starCount = 0;
            for (Reply reply : replyList) {
                if (reply.getAnalysisRoundExamTitleId().intValue() == examTitleIdList.get(idx).intValue()) {
                    isReplied =  true;
                    starCount = reply.getStarCount();
                    break;
                }
            }
            userExamTitleDto.setIsReplied(isReplied ? 1 : 0);
            userExamTitleDto.setStarCount(starCount);

            userExamTitleDtos.add(userExamTitleDto);
        }

        return userExamTitleDtos;
    }

    Integer getReplyStatus(CHCAnalysis analysis, Integer examTitleId,  Integer userId) {
        Doctor doctor = doctorService.getDoctorById(userId);

        if (doctor != null) {
            if (analysis.getType() == ANALYSIS_TYPE_CHC) {
                /**
                 * CHC 获取考试前两百名
                 */
                if (chcRankingListService.isRankingInTop200(doctor.getId())) {
                    return getReplyStatus(examTitleId, userId);
                }

            } else if (analysis.getType() == ANALYSIS_TYPE_PSA) {
                /**
                 * 如果未认证 不可见
                 */
                if (!(doctor.getStatus().equals(CommonConstants.DOCTOR_STATUS_3) ||
                        doctor.getStatus().equals(CommonConstants.DOCTOR_STATUS_6) ||
                        doctor.getStatus().equals(CommonConstants.DOCTOR_STATUS_7) )) {
                    return 0;
                }
                /**
                 * PSA 职务职称主任/副主任 院长/副院长 科长/副科长 管理人数100以上
                 */
                if (doctorService.isAuth(doctor.getId())) {
                    return getReplyStatus(examTitleId, userId);
                }
            }
        }

        return 0;
    }
}
