package com.pica.cloud.permission.permission.aop;

import com.pica.cloud.foundation.entity.PicaException;
import com.pica.cloud.foundation.entity.PicaResponse;
import com.pica.cloud.foundation.entity.PicaResultCode;
import com.pica.cloud.foundation.redis.ICacheClient;
import com.pica.cloud.permission.permission.client.PermissionServiceClient;
import com.pica.cloud.permission.permission.common.annotation.Authentication;
import com.pica.cloud.permission.permission.common.constants.AuthTypeEnum;
import com.pica.cloud.permission.permission.common.constants.PermissionResultCode;
import com.pica.cloud.permission.permission.common.constants.ProductTypeEnum;
import com.pica.cloud.permission.permission.common.dto.AuthDto;
import com.pica.cloud.permission.permission.common.dto.AuthResultDto;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import javax.servlet.http.HttpServletRequest;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;

/**
 * @author andong
 * @create 2019/9/11
 */
@Aspect
@Component
public class AuthAspect {

    private final String TOKEN_PREFIX = "token-";
    private final String DOCTOR_TOKEN_PREFIX = "token-doctor-";
    private final Map<Method, Authentication> methodAuthMap = new HashMap();
    private final Map<Method, String> methodUrlMap = new HashMap();
    private final Map<Method, Integer> methodIndexMap = new HashMap();

    @Autowired
    private ICacheClient cacheClient;
    @Autowired
    private PermissionServiceClient permissionServiceClient;

    @Pointcut("@annotation(com.pica.cloud.permission.permission.common.annotation.Authentication)")
    public void auth() {}

    @Around("auth()")
    public Object permissionAuth(ProceedingJoinPoint joinPoint) throws Throwable {
        //获取鉴权注解配置信息
        MethodSignature sign = (MethodSignature) joinPoint.getSignature();
        Method method = sign.getMethod();
        Authentication authentication = methodAuthMap.get(method);
        if (authentication == null) {
            authentication = method.getAnnotation(Authentication.class);
            methodAuthMap.put(method, authentication);
        }

        //根据token获取userId
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();
        String token = request.getHeader("token");
        //是否需要校验token
        if ((token == null || "".equals(token.trim())) && !authentication.tokenRequired()) {
            return joinPoint.proceed();
        }
        int userId = 0;
        try {
            if (authentication.productType() == ProductTypeEnum.DOCTOR.code()) {  //获取云鹊医doctorId
                String tokenValue = cacheClient.get(TOKEN_PREFIX + token);
                userId = Integer.valueOf(tokenValue.replace(DOCTOR_TOKEN_PREFIX, ""));
            }
            //TODO
        } catch (Exception ex) {
        }
        if (userId == 0) {  //未登录
            throw new PicaException(PicaResultCode.LOGIN_FAILE.code(), PicaResultCode.LOGIN_FAILE.message());
        }

        //判断鉴权类型
        List<String> roleCodes = new ArrayList();
        int authType = AuthTypeEnum.URL.code();
        if (authentication.roleCodes().length > 0) {
            authType = AuthTypeEnum.ROLE.code();
            roleCodes = Arrays.asList(authentication.roleCodes());
        }

        //url鉴权，获取api
        String api = null;
        if (authType == AuthTypeEnum.URL.code()) {
            api = methodUrlMap.get(method);
            if (api == null) {
                String requestMappingUrl = this.getRequestMappingUrl(joinPoint.getTarget(), method);
                api = request.getMethod() + ":" + request.getContextPath() + requestMappingUrl;
                methodUrlMap.put(method, api);
            }
        }

        //访问权限中心进行鉴权
        AuthDto authDto = new AuthDto(authentication.productType(), userId, authType, roleCodes, api, authentication.dataAuth(), null);
        PicaResponse<AuthResultDto> result = permissionServiceClient.auth(authDto);
        if (!PicaResultCode.SUCCESS.code().equals(result.getCode())) {
            throw new PicaException(result.getCode(), result.getMessage());
        }
        AuthResultDto authResult = result.getData();
        if (!authResult.isAccess()) {  //无操作权限
            throw new PicaException(PermissionResultCode.PERMISSION_DENY.code(), PermissionResultCode.PERMISSION_DENY.message());
        }

        //设置数据权限信息
        if (authentication.dataAuth()) {
            //获取方法签名上AuthResultDto类型参数的位置信息
            Integer index = methodIndexMap.get(method);
            if (index == null) {
                index = -1;
                Class[] types = method.getParameterTypes();
                if (types != null && types.length > 0) {
                    for (int i = 0; i < types.length; i++) {
                        if (types[i] == AuthResultDto.class) {
                            index = i;
                            break;
                        }
                    }
                }
                methodIndexMap.put(method, index);
            }

            if (index >= 0) {
                Object[] objs = joinPoint.getArgs();
                objs[index] =  authResult;
                return joinPoint.proceed(objs);
            }
        }

        return joinPoint.proceed();
    }

    //获取Controller+Method上配置的uri信息
    private String getRequestMappingUrl(Object controller, Method method) {
        RequestMapping classUrl = controller.getClass().getAnnotation(RequestMapping.class);
        String url = "";
        if (classUrl != null) {
            url += classUrl.value()[0];  //Controller类上RequestMapping配置
        }

        //方法上RequestMapping配置
        Annotation[] anns = method.getAnnotations();
        for (Annotation ann : anns) {
            if (ann.annotationType() == RequestMapping.class) {
                url += ((RequestMapping) ann).value()[0];
                break;
            }
            if (ann.annotationType() == GetMapping.class) {
                url += ((GetMapping) ann).value()[0];
                break;
            }
            if (ann.annotationType() == PostMapping.class) {
                url += ((PostMapping) ann).value()[0];
                break;
            }
            if (ann.annotationType() == PutMapping.class) {
                url += ((PutMapping) ann).value()[0];
                break;
            }
            if (ann.annotationType() == DeleteMapping.class) {
                url += ((DeleteMapping) ann).value()[0];
                break;
            }
            if (ann.annotationType() == PatchMapping.class) {
                url += ((PatchMapping) ann).value()[0];
                break;
            }
        }
        return url;
    }
}
