package com.session

import java.sql
import java.sql.PreparedStatement
import com.config.MyConfigSession
import com.pica.utils.DateUtils
import com.utils.{JDBCUtil, UseUtil}
import org.apache.spark.SparkConf
import org.apache.spark.sql.expressions.{Window, WindowSpec}
import org.apache.spark.sql.functions.{lag, row_number}
import org.apache.spark.sql.{DataFrame, SparkSession}


/**
  * 处理昨天的数据,导入到pica_dw.dw_fact_log_session_path表
  * @Author zhenxin.ma
  * @Date 2020/3/27 10:58
  * @Version 1.0
  */
class SessionProcessPath {
    def getSparkSession(appName: String): SparkSession = {
        val conf: SparkConf = new SparkConf().setAppName(appName)
        UseUtil.setConfigure(conf)
        val sparkSession: SparkSession = SparkSession.builder().config(conf).enableHiveSupport().getOrCreate()
        sparkSession
    }
}


object SessionProcessPath {
    def apply(): SessionProcessPath = new SessionProcessPath()


    def main(args: Array[String]): Unit = {
        //1.执行任务之前先往record表记录
        val insertSQL: String =
            s"""
               			   |insert into ${MyConfigSession.DATA_BASE}.${MyConfigSession.JDBC_TABLE} (job_id,job_name,job_type,job_scn,status,start_time)
               			   |values(1968,'pica_dw.dw_fact_log_session_path','3',?,'0',?)
		""".stripMargin
        //设置同步数据的批次号,格式是2019-09-12
        val scnData: String = DateUtils.getYesterdayDate
        //设置任务开始时间,格式是2019-09-12 14:03:30
        val startTime: String = DateUtils.getTodayTime
        //存储SQL中的参数
        val insertArr: Array[String] = Array[String](scnData, startTime)
        //获取MYSQL连接
        val connSql: sql.Connection = JDBCUtil.getConnection()
        //向 record 表插入数据
        val flag: Int = JDBCUtil.insertRecord(connSql, insertSQL, insertArr)
        try {
            val sparkSession: SparkSession = SessionProcessPath().getSparkSession("SessionProcessPath")
            //筛选源数据
            val sourceDF: DataFrame = sparkSession.sql(MyConfigSession.SOURCE_SQL_PATH)

            //注册日期在流量统计日期之前的用户
            val doctorDF: DataFrame = sparkSession.sql(
                "select cast(id as string) id from pica_ds.pica_doctor where to_date(creat_time) <=DATE_SUB(current_date(),1)")

            sourceDF.join(doctorDF, sourceDF("user_id") === doctorDF("id"), "left")
                    .createOrReplaceTempView("tmp_table")

            //将id为null的记录设置为0
            val reSql: String = "select session_id,case when id is null then '0' else user_id END as user_id,action_type," +
                    "user_token,menu_code,action_code,position,label_value,app_version,device_type,created_time,date_time from tmp_table"
            val selectDF: DataFrame = sparkSession.sql(reSql)


            println("-----------------------------------compute refer columns-----------------------------------------")
            val resultDF: DataFrame = getReferColumns(selectDF,sparkSession)

            println("-----------------------------------load data to pica_dw.dw_fact_log_session_path-----------------")
            loadData(resultDF,sparkSession,scnData)

            println("----------------------------------update task record table---------------------------------------")
            //任务执行成功,更新 Mysql record 配置表
            val updateSQL: String =
                s"""
                   |update ${MyConfigSession.JDBC_TABLE} set status=?,end_time=?,data_count=? where job_id=1968 and start_time='${startTime}'
				""".stripMargin
            val upreSta: PreparedStatement = connSql.prepareStatement(updateSQL)
            upreSta.setString(1, "1")
            upreSta.setString(2, DateUtils.getTodayTime)
            upreSta.setInt(3, resultDF.count().toInt)
            //更新表数据
            upreSta.executeUpdate()
            //关闭连接
            JDBCUtil.close(connSql, upreSta)
            sparkSession.stop()
        }catch {
            case e:Exception => {
                println("-----------------------------------任务异常---------------------------------------------------")
                val exceptionSQL: String =
                    s"""
                       |update ${MyConfigSession.JDBC_TABLE} set status=?,exception=?,end_time=? where job_id=1968 and start_time='${startTime}'
					""".stripMargin
                val errorArr = Array[String]("2", e.getMessage, DateUtils.getTodayTime)
                JDBCUtil.insertRecord(connSql, exceptionSQL, errorArr)
                connSql.close()
            }

        }
    }


    /**
      * @Description         获取需要的字段的refer字段
      * @param dataFrame     源数据
      * @param sparkSession  SparkSession 环境
      * @return org.apache.spark.sql.Dataset<org.apache.spark.sql.Row>
      **/
    def getReferColumns(dataFrame: DataFrame ,sparkSession: SparkSession):DataFrame = {
        //先按照 session_id分区,再按照 created_time排序,进行窗口计算
        val sessionIDWinSpec: WindowSpec = Window.partitionBy("session_id").orderBy("created_time")
        //增叫refer_字段
        val menuDF: DataFrame =
            dataFrame.withColumn("refer_menu_code", lag(dataFrame("menu_code"), 1).over(sessionIDWinSpec))
        val acodeDF: DataFrame =
            menuDF.withColumn("refer_action_code", lag(menuDF("action_code"), 1).over(sessionIDWinSpec))
        val positionDF: DataFrame =
            acodeDF.withColumn("refer_position", lag(acodeDF("position"), 1).over(sessionIDWinSpec))
        val actypeDF: DataFrame =
            positionDF.withColumn("refer_action_type", lag(positionDF("action_type"), 1).over(sessionIDWinSpec))
        val recreatDF: DataFrame =
            actypeDF.withColumn("refer_created", lag(actypeDF("created_time"), 1).over(sessionIDWinSpec))
        val rowNumberDF: DataFrame =
            recreatDF.withColumn("step_id", row_number().over(sessionIDWinSpec))

        //去掉refer字段中的NULL值
        val coaleseDF: DataFrame = rowNumberDF.selectExpr(
            "session_id", "user_id", "action_type", "user_token", "menu_code", "action_code", "position", "label_value",
            "COALESCE(refer_menu_code,'') as refer_menu_code",
            "COALESCE(refer_action_code,'') as refer_action_code",
            "COALESCE(refer_position,'') as refer_position",
            "COALESCE(refer_action_type,'') as refer_action_type",
            "COALESCE(refer_created,created_time) as refer_created",
            "step_id", "app_version", "device_type", "created_time", "date_time")

        //在此基础上增加字段 refer_time_diff,值为 created_time, refer_created 之差
        val referTimeDiff: DataFrame =
            coaleseDF.withColumn("refer_time_diff", coaleseDF("created_time") - coaleseDF("refer_created"))
        referTimeDiff
    }



    /**
      * @Description         导入数据到表中
      * @param dataFrame     源数据
      * @param sparkSession  SparkSession 环境
      * @param partitionDay  分区日期
      * @return void
      **/
    def loadData(dataFrame: DataFrame, sparkSession: SparkSession, partitionDay:String):Unit = {
        dataFrame.createOrReplaceTempView("result_view")
        val loadDataSql =
            s"""
               |insert overwrite table ${MyConfigSession.HIVE_TABLE2} partition(created_day='${partitionDay}')
               | select session_id,user_id,action_type,user_token,menu_code,action_code,position,label_value,
               | refer_menu_code,refer_action_code,refer_position,refer_action_type,
               | cast(refer_time_diff as int) as refer_time_diff,step_id,app_version,device_type,created_time,date_time
               | from result_view
                """.stripMargin
        sparkSession.sql(loadDataSql)
    }


}
